#pragma once

#include <library/cpp/regex/hyperscan/hyperscan.h>

namespace NHyperscan {

enum EPlatform {
    ANY,
    AVX2,
    AVX512
};

using TDatabasePtr = hs_database_t*;

template<typename TCallback>
void ScanPtr(
    TDatabasePtr db,
    const TScratch& scratch,
    const TStringBuf& text,
    TCallback& callback // applied to index of matched regex
) {
    struct TCallbackWrapper {
        static int EventHandler(
            unsigned int id,
            unsigned long long from,
            unsigned long long to,
            unsigned int flags,
            void* ctx) {
            Y_UNUSED(flags);
            TCallback& callback2 = *reinterpret_cast<TCallback*>(ctx);
            callback2(id, from, to);
            return 0;
        }
    };
    unsigned int flags = 0; // unused at present
    hs_error_t status = Singleton<NPrivate::TImpl>()->Scan(
        db,
        text.begin(),
        static_cast<unsigned int>(text.size()),
        flags,
        scratch.Get(),
        &TCallbackWrapper::EventHandler,
        &callback);
    if (status != HS_SUCCESS) {
        ythrow yexception() << "Failed to scan against text: " << text;
    }
}

bool MatchesPtr(
    TDatabasePtr db,
    const TScratch& scratch,
    const TStringBuf& text
);

TDatabase CompileFor(
    const TStringBuf& regex,
    unsigned int flags,
    EPlatform platform
);

TDatabase CompileMultiFor(
    const TVector<const char*>& regexs,
    const TVector<unsigned int>& flags,
    const TVector<unsigned int>& ids,
    EPlatform platform
);

inline EPlatform CurrentPlatform() {
    if (NX86::HaveAVX512F() && NX86::HaveAVX512BW()) {
        return EPlatform::AVX512;
    } else if (NX86::HaveAVX() && NX86::HaveAVX2()) {
        return EPlatform::AVX2;
    } else {
        return EPlatform::ANY;
    }
}

}  // namespace NHyperscan
