package acler

import (
	"errors"
	"os"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/libs/go/acler/staff"
)

const (
	aclUpdateDuration = 86400 * time.Second
)

type ACLer struct {
	options       *Options
	allowedLogins map[string]bool
	allowedTVMs   map[tvm.ClientID]bool
	aclLock       sync.RWMutex
	anyUsers      bool
}

func New(options ...Option) (*ACLer, error) {
	opts := &Options{
		StaffToken: os.Getenv("STAFF_TOKEN"),
	}

	for _, opt := range options {
		opt(opts)
	}

	if (len(opts.Departments) > 0 || len(opts.ABCRoles) > 0) && opts.StaffToken == "" {
		return nil, errors.New("staff groups doesn't works w/o staff token")
	}

	result := &ACLer{
		options:       opts,
		allowedLogins: make(map[string]bool, len(opts.Users)),
		allowedTVMs:   make(map[tvm.ClientID]bool, len(opts.TVMs)),
	}

	for _, tvmID := range opts.TVMs {
		result.allowedTVMs[tvmID] = true
	}

	result.RefreshACL()
	return result, nil
}

func (a *ACLer) RefreshACL() {
	acl := map[string]bool{}
	for _, login := range a.options.Users {
		acl[login] = true
	}

	if logins, err := a.fetchDepsLogins(a.options.Departments); err == nil {
		for _, login := range logins {
			acl[login] = true
		}
	}

	if logins, err := a.fetchAbcLogins(a.options.ABCRoles); err == nil {
		for _, login := range logins {
			acl[login] = true
		}
	}

	a.aclLock.Lock()
	a.allowedLogins = acl
	a.aclLock.Unlock()
}

func (a *ACLer) AutoRefresh() {
	go a.RefreshGroupsLoop()
}

func (a *ACLer) RefreshGroupsLoop() {
	for {
		a.RefreshACL()
		time.Sleep(aclUpdateDuration)
	}
}

func (a *ACLer) CheckUser(login string) bool {
	a.aclLock.RLock()
	_, validUser := a.allowedLogins[login]
	a.aclLock.RUnlock()
	return validUser
}

func (a *ACLer) CheckTvm(tvmID tvm.ClientID) bool {
	_, ok := a.allowedTVMs[tvmID]
	return ok
}

func (a *ACLer) fetchDepsLogins(deps []string) (logins []string, err error) {
	if len(deps) == 0 {
		return
	}

	for _, dep := range deps {
		allowedLogins, err := staff.ResolveDepartment(a.options.StaffToken, dep)
		if err != nil {
			return logins, err
		}

		logins = append(logins, allowedLogins...)
	}

	return
}

func (a *ACLer) fetchAbcLogins(roles []string) (logins []string, err error) {
	if len(roles) == 0 {
		return
	}

	for _, role := range roles {
		allowedLogins, err := staff.ResolveABCRole(a.options.StaffToken, role)
		if err != nil {
			return logins, err
		}

		logins = append(logins, allowedLogins...)
	}

	return
}
