package archer

import (
	"errors"
	"io"
	"strings"
)

const (
	ExtTarBz2 = ".tar.bz2"
	ExtTarGz  = ".tar.gz"
	ExtTar    = ".tar"
	ExtTgz    = ".tgz"
	ExtTbz    = ".tbz"
	ExtTbz2   = ".tbz2"
	ExtZip    = ".zip"
	ExtWhl    = ".whl"
	ExtEgg    = ".egg"
	ExtJar    = ".jar"
	ExtWar    = ".war"
)

type (
	WalkPattern struct {
		ID      int
		Marker  string
		Pattern string
	}

	FileWalkOpts struct {
		// Once - walk only once
		Once bool

		// Patterns
		Patterns []WalkPattern
	}

	FileWalkFunc func(targetPath string, patternID int, reader SizeReader) error

	Walker interface {
		FileWalk(src string, opts FileWalkOpts, walkFn FileWalkFunc) error
		Walk(r SizeReader, opts FileWalkOpts, walkFn FileWalkFunc) error
	}
)

var (
	// Walkers is the mapping of extension to the Walker that will walk that extension/type
	Walkers map[string]Walker

	// ErrStop - error returns by WalkFunc to stop iteration
	ErrStop = errors.New("stop")

	// ErrUnknownArchive - archer doesn't work with this format
	ErrUnknownArchive = errors.New("unknown archive format")
)

func init() {
	tbzWalker := new(TarBzip2Walker)
	tgzWalker := new(TarGzipWalker)
	zipWalker := new(ZipWalker)

	Walkers = map[string]Walker{
		ExtTarBz2: tbzWalker,
		ExtTarGz:  tgzWalker,
		ExtTar:    new(TarWalker),
		ExtTgz:    tgzWalker,
		ExtTbz:    tbzWalker,
		ExtTbz2:   tbzWalker,
		ExtZip:    zipWalker,
		ExtWhl:    zipWalker,
		ExtEgg:    zipWalker,
		ExtJar:    zipWalker,
		ExtWar:    zipWalker,
	}
}

func WalkerFor(filename string) (Walker, error) {
	ext := "."
	parts := strings.Split(filename, ".")
	if len(parts) > 1 && parts[len(parts)-2] == "tar" {
		ext += strings.Join(parts[len(parts)-2:], ".")
	} else {
		ext += parts[len(parts)-1]
	}

	walker, ok := Walkers[ext]
	if !ok {
		return nil, ErrUnknownArchive
	}
	return walker, nil
}

func silenceClose(r io.Closer) {
	_ = r.Close()
}
