package archer

import (
	"archive/tar"
	"io"
	"path"
	"strings"
)

type TarWalker struct{}

func tarWalk(input io.Reader, opts FileWalkOpts, walkFn FileWalkFunc) error {
	expectedCalls := 0
	var handled map[string]bool
	if opts.Once {
		expectedCalls = len(opts.Patterns)
		handled = make(map[string]bool, expectedCalls)
		for _, p := range opts.Patterns {
			handled[p.Pattern] = false
		}
	}

	calls := 0
	tarR := tar.NewReader(input)
loop:
	for {
		hdr, err := tarR.Next()
		if err == io.EOF {
			return nil
		}

		if err != nil {
			return err
		}

		if hdr.Typeflag != tar.TypeReg {
			// Not a file
			continue
		}

		for _, p := range opts.Patterns {
			if !strings.Contains(hdr.Name, p.Marker) {
				continue
			}

			if matched, _ := path.Match(p.Pattern, hdr.Name); !matched {
				continue
			}

			if handled != nil {
				if handled[p.Pattern] {
					continue loop
				}
				handled[p.Pattern] = true
			}

			reader, err := OpenReader(tarR, hdr.Size)
			if err != nil {
				return err
			}

			if err := walkFn(hdr.Name, p.ID, reader); err != nil {
				_ = reader.Close()
				if err == ErrStop {
					return nil
				}

				return err
			}

			_ = reader.Close()
			calls++
			if opts.Once && calls >= expectedCalls {
				// Done!
				break
			}
			continue loop
		}
	}
}

func (z *TarWalker) FileWalk(src string, opts FileWalkOpts, walkFn FileWalkFunc) error {
	f, err := OpenFile(src)
	if err != nil {
		return err
	}
	defer silenceClose(f)

	return z.Walk(f, opts, walkFn)
}

func (z *TarWalker) Walk(r SizeReader, opts FileWalkOpts, walkFn FileWalkFunc) error {
	return tarWalk(r, opts, walkFn)
}
