package archer

import (
	"archive/zip"
	"os"
	"path"
	"strings"
)

type ZipWalker struct{}

func zipWalk(input *zip.Reader, opts FileWalkOpts, walkFn FileWalkFunc) error {
	expectedCalls := 0
	var handled map[string]bool
	if opts.Once {
		expectedCalls = len(opts.Patterns)
		handled = make(map[string]bool, expectedCalls)
		for _, p := range opts.Patterns {
			handled[p.Pattern] = false
		}
	}

	calls := 0
loop:
	for _, f := range input.File {
		if f.FileInfo().Mode()&os.ModeType != 0 {
			// Not a file
			continue
		}

		for _, p := range opts.Patterns {
			if !strings.Contains(f.Name, p.Marker) {
				continue
			}

			if matched, _ := path.Match(p.Pattern, f.Name); !matched {
				continue
			}

			if handled != nil {
				if handled[p.Pattern] {
					continue loop
				}
				handled[p.Pattern] = true
			}

			reader, err := f.Open()
			if err != nil {
				return err
			}

			sReader, err := OpenReader(reader, f.FileInfo().Size())
			if err != nil {
				_ = reader.Close()
				return err
			}

			if err := walkFn(f.Name, p.ID, sReader); err != nil {
				_ = sReader.Close()
				_ = reader.Close()

				if err == ErrStop {
					return nil
				}

				return err
			}

			_ = sReader.Close()
			_ = reader.Close()

			calls++
			if opts.Once && calls >= expectedCalls {
				// Done!
				break
			}
			continue loop
		}
	}

	return nil
}

func (z *ZipWalker) FileWalk(src string, opts FileWalkOpts, walkFn FileWalkFunc) error {
	r, err := OpenFile(src)
	if err != nil {
		return err
	}

	defer silenceClose(r)
	return z.Walk(r, opts, walkFn)
}

func (z *ZipWalker) Walk(r SizeReader, opts FileWalkOpts, walkFn FileWalkFunc) error {
	zipR, err := zip.NewReader(r, r.Size())
	if err != nil {
		return err
	}

	return zipWalk(zipR, opts, walkFn)
}
