package main

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/libs/go/boombox/httpreplay"
	"a.yandex-team.ru/security/libs/go/boombox/tape"
)

func main() {
	var (
		tapePath       string
		clearTape      bool
		namespace      string
		replyMode      bool
		addr           string
		upstream       string
		followRedirect bool
	)

	pflag.BoolVar(&replyMode, "reply", false, "start's boombox in reply mode")
	pflag.StringVar(&tapePath, "tape-path", "boombox.bolt", "boombox tape file path")
	pflag.BoolVar(&clearTape, "new-clear", false, "create new tape (not available with --reply)")
	pflag.StringVar(&namespace, "namespace", httpreplay.DefaultNamespace, "boombox namespace")
	pflag.StringVar(&addr, "addr", "localhost:3000", "listen addr")
	pflag.StringVar(&upstream, "upstream", "", "upstream url (for proxy mode only)")
	pflag.BoolVar(&followRedirect, "follow-redirect", false, "follow upstream response redirect")
	pflag.Parse()

	l, err := zap.New(zap.ConsoleConfig(log.DebugLevel))
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "can't create logger: %v\n", err)
		os.Exit(1)
	}

	recOpts := []httpreplay.Option{
		httpreplay.WithLogger(l),
		httpreplay.WithNamespace(namespace),
		httpreplay.WithFollowRedirect(followRedirect),
	}

	if !replyMode {
		if clearTape {
			// clear current cache
			_ = os.Remove(tapePath)
		}

		recOpts = append(recOpts, httpreplay.WithProxyMode(upstream))
	}

	s, err := tape.NewTape(tapePath)
	if err != nil {
		l.Fatalf("can't create storage: %v", err)
		return
	}

	recorder, err := httpreplay.NewReplay(s, recOpts...)
	if err != nil {
		l.Fatalf("can't create recorder: %v", err)
		return
	}

	errChan := make(chan error, 1)
	okChan := make(chan struct{}, 1)
	go func() {
		if err := recorder.ListenAndServe(addr); err != nil {
			errChan <- err
		} else {
			okChan <- struct{}{}
		}
	}()

	stopChan := make(chan os.Signal, 2)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	select {
	case <-stopChan:
		l.Info("shutdown signal received")

		ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
		defer cancel()

		if err := recorder.Shutdown(ctx); err != nil {
			l.Errorf("failed to gracefully shutdown recorder: %v", err)
		}

		if err := s.Close(); err != nil {
			l.Errorf("failed to gracefully close storage: %v", err)
		}
	case <-okChan:
	case err := <-errChan:
		l.Errorf("failed to start recorder: %v", err)
	}
}
