package httpmodel

import (
	"bytes"
	"context"
	"io"
	"net/http"
)

func RequestFromHTTP(req *http.Request) (*Request, error) {
	out := &Request{
		Method:  req.Method,
		Url:     req.RequestURI,
		Headers: make([]*Header, 0, len(req.Header)),
	}

	for h, vv := range req.Header {
		out.Headers = append(out.Headers, &Header{
			Name:   h,
			Values: vv,
		})
	}

	if req.Body != nil {
		defer func() {
			_ = req.Body.Close()
		}()
		var err error
		out.Body, err = io.ReadAll(req.Body)
		if err != nil {
			return nil, err
		}
	}

	return out, nil
}

func (x *Request) NewHTTPRequest(ctx context.Context) (*http.Request, error) {
	r, err := http.NewRequestWithContext(ctx, x.Method, x.Url, nil)
	if err != nil {
		return nil, err
	}

	if len(x.Body) > 0 {
		r.Body = io.NopCloser(bytes.NewReader(x.Body))
	}

	copyHeader(r.Header, x.Headers)
	return r, nil
}
