package httpmodel

import (
	"bytes"
	"io"
	"net/http"

	"google.golang.org/protobuf/types/known/timestamppb"
)

func ResponseFromHTTP(rsp *http.Response) (*Response, error) {
	out := &Response{
		UpdatedAt:  timestamppb.Now(),
		StatusCode: int32(rsp.StatusCode),
		Headers:    make([]*Header, 0, len(rsp.Header)),
	}

	for h, vv := range rsp.Header {
		out.Headers = append(out.Headers, &Header{
			Name:   h,
			Values: vv,
		})
	}

	if rsp.StatusCode >= 300 && rsp.StatusCode < 400 {
		if l := rsp.Header.Get("Location"); l == "" {
			// recreate location header
			out.Headers = append(out.Headers, &Header{
				Name:   "Location",
				Values: []string{rsp.Request.URL.String()},
			})
		}
	}

	var err error
	out.Body, err = io.ReadAll(rsp.Body)
	if err != nil {
		return nil, err
	}
	return out, nil
}

func (x *Response) WriteTo(w http.ResponseWriter) error {
	copyHeader(w.Header(), x.Headers)
	w.WriteHeader(int(x.StatusCode))
	_, err := io.Copy(w, bytes.NewReader(x.Body))
	return err
}

func copyHeader(dst http.Header, src []*Header) {
	for _, h := range src {
		for _, v := range h.Values {
			dst.Add(h.Name, v)
		}
	}
}
