package main

import (
	"encoding/gob"
	"errors"
	"fmt"
	"net/http/httptest"
	"os"
	"os/exec"
	"os/signal"
	"strings"
	"syscall"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/test/recipe"
	"a.yandex-team.ru/library/go/test/yatest"
	"a.yandex-team.ru/security/libs/go/boombox/httpreplay"
	"a.yandex-team.ru/security/libs/go/boombox/tape"
)

var (
	tapePath  string
	namespace string
	isChild   bool
)

var _ recipe.Recipe = (*boomboxRecipe)(nil)

type boomboxRecipe struct{}

type boomboxInfo struct {
	URL string
	PID int
}

func (fp boomboxRecipe) Start() error {
	if tapePath == "" || tapePath == "unset" {
		return errors.New("${BOOMBOX_DBPATH} is unset")
	}

	if namespace == "" || namespace == "unset" {
		return errors.New("${BOOMBOX_NAMESPACE} is unset")
	}

	if isChild {
		// we are in daemonized child
		zapCFG := zap.ConsoleConfig(log.DebugLevel)
		zapCFG.OutputPaths = []string{
			yatest.OutputPath(fmt.Sprintf("boombox-%s.log", namespace)),
		}
		l, err := zap.New(zapCFG)
		if err != nil {
			return fmt.Errorf("can't create logger: %w", err)
		}

		srv, err := newServer(l)
		if err != nil {
			l.Error("can't start server", log.Error(err))
			return err
		}

		err = writeBoomboxInfo(&boomboxInfo{
			URL: srv.URL,
			PID: os.Getpid(),
		})
		if err != nil {
			srv.Close()
			l.Error("can't save boombox info", log.Error(err))
			return err
		}

		l.Info("server started")
		stopChan := make(chan os.Signal, 1)
		signal.Notify(stopChan, syscall.SIGINT)

		<-stopChan
		srv.Close()
		return nil
	}

	l, err := zap.New(zap.ConsoleConfig(log.DebugLevel))
	if err != nil {
		return fmt.Errorf("can't create logger: %w", err)
	}

	bi, err := startChild(l)
	if err != nil {
		return err
	}

	envName := func(key string) string {
		return strings.ToUpper(fmt.Sprintf("BOOMBOX_%s_%s", namespace, key))
	}

	recipe.SetEnv(envName("URL"), bi.URL)
	recipe.SetEnv(envName("PID"), fmt.Sprint(bi.PID))
	return nil
}

func (fp boomboxRecipe) Stop() error {
	if _, err := os.Stat(infoFilename()); err != nil {
		if os.IsNotExist(err) {
			// that's fine
			return nil
		}
		return err
	}

	bi, err := readBoomboxInfo()
	if err != nil {
		return err
	}

	return syscall.Kill(bi.PID, syscall.SIGINT)
}

func startChild(l log.Logger) (*boomboxInfo, error) {
	_ = os.Remove(infoFilename())
	errs := make(chan error, 1)
	out := make(chan *boomboxInfo, 1)
	go func() {
		path, err := os.Executable()
		if err != nil {
			errs <- fmt.Errorf("failed to get currect executable: %w", err)
			return
		}

		cmd := exec.Command(path, append(os.Args[1:], "--boombox-is-a-child")...)
		if err = cmd.Start(); err != nil {
			errs <- err
			return
		}

		waiter := func() error {
			pi, err := readBoomboxInfo()
			if err != nil {
				return err
			}

			out <- pi
			return nil
		}

		notify := func(err error, _ time.Duration) {
			l.Warn("wait boombox srv fail", log.Error(err))
		}

		b := backoff.WithMaxRetries(backoff.NewConstantBackOff(1*time.Second), 10)
		err = backoff.RetryNotify(waiter, b, notify)
		if err != nil {
			errs <- err
		}
	}()

	select {
	case err := <-errs:
		return nil, fmt.Errorf("failed to start child: %w", err)
	case pi := <-out:
		return pi, nil
	}
}

func newServer(l log.Logger) (*httptest.Server, error) {
	s, err := tape.NewTape(tapePath, tape.WithReadOnly())
	if err != nil {
		return nil, fmt.Errorf("can't create boombox storage: %w", err)
	}

	r, err := httpreplay.NewReplay(s, httpreplay.WithLogger(l), httpreplay.WithNamespace(namespace))
	if err != nil {
		return nil, fmt.Errorf("can't create boombox records replay: %w", err)
	}

	return httptest.NewServer(r), nil
}

func readBoomboxInfo() (*boomboxInfo, error) {
	f, err := os.Open(infoFilename())
	if err != nil {
		return nil, err
	}
	defer func() { _ = f.Close() }()

	var bi boomboxInfo
	err = gob.NewDecoder(f).Decode(&bi)
	if err != nil {
		return nil, err
	}
	return &bi, nil
}

func writeBoomboxInfo(bi *boomboxInfo) error {
	f, err := os.Create(infoFilename())
	if err != nil {
		return err
	}
	defer func() { _ = f.Close() }()

	return gob.NewEncoder(f).Encode(bi)
}

func infoFilename() string {
	return fmt.Sprintf("boombox-%s.status.gob", namespace)
}

func main() {
	pflag.BoolVar(&isChild, "boombox-is-a-child", false, "")
	pflag.StringVar(&tapePath, "boombox-tape-path", "unset", "boombox tape path")
	pflag.StringVar(&namespace, "boombox-namespace", "unset", "boombox namespace")
	recipe.Run(boomboxRecipe{})
}
