package tape

import (
	"errors"
	"fmt"
	"time"

	"go.etcd.io/bbolt"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/security/libs/go/boombox/httpmodel"
)

var (
	ErrRecordNotFound = errors.New("record not found")
)

type Tape struct {
	db       *bbolt.DB
	boltOpts *bbolt.Options
}

func NewTape(tapePath string, opts ...Option) (*Tape, error) {
	tape := &Tape{
		boltOpts: &bbolt.Options{
			Timeout: 1 * time.Second,
		},
	}

	for _, opt := range opts {
		opt(tape)
	}

	var err error
	tape.db, err = bbolt.Open(tapePath, 0644, tape.boltOpts)
	if err != nil {
		return nil, fmt.Errorf("failed to open bbolt db: %w", err)
	}

	return tape, nil
}

func (t *Tape) GetHTTPResponse(bucket, key string) (*httpmodel.Response, error) {
	var rsp httpmodel.Response
	return &rsp, t.Get(bucket, key, &rsp)
}

func (t *Tape) SaveHTTPResponse(bucket, key string, rsp *httpmodel.Response) error {
	return t.Save(bucket, key, rsp)
}

func (t *Tape) Get(bucket, key string, dst proto.Message) error {
	return t.db.View(func(tx *bbolt.Tx) error {
		bucket := tx.Bucket([]byte(bucket))
		if bucket == nil {
			return bbolt.ErrBucketNotFound
		}

		b := bucket.Get([]byte(key))
		if b == nil {
			return ErrRecordNotFound
		}

		if err := proto.Unmarshal(b, dst); err != nil {
			return fmt.Errorf("failed to decode item: %w", err)
		}

		return nil
	})
}

func (t *Tape) Save(bucket, key string, in proto.Message) error {
	return t.db.Update(func(tx *bbolt.Tx) error {
		bucket, err := tx.CreateBucketIfNotExists([]byte(bucket))
		if err != nil {
			return err
		}

		in, err := proto.Marshal(in)
		if err != nil {
			return fmt.Errorf("failed to encode item: %w", err)
		}

		return bucket.Put([]byte(key), in)
	})
}

func (t *Tape) Close() error {
	return t.db.Close()
}
