package daemon

import (
	"encoding/json"
	"errors"
	"fmt"
)

type ChildNotificationKind int

const (
	ChildNotificationKindNone = iota
	ChildNotificationKindStarted
	ChildNotificationKindError
)

type ChildNotification struct {
	Kind ChildNotificationKind
	Data json.RawMessage
}

func (n *ChildNotification) Error() error {
	if n.Kind != ChildNotificationKindError {
		return nil
	}

	var out string
	if err := n.parseData(&out); err != nil {
		return fmt.Errorf("child failed to start, but we can't parse it: %w", err)
	}

	return errors.New(out)
}

func (n *ChildNotification) ParseData(dst interface{}) error {
	if n.Kind != ChildNotificationKindStarted {
		return fmt.Errorf("unexpected child message kind: %d", n.Kind)
	}

	return n.parseData(dst)
}

func (n *ChildNotification) parseData(dst interface{}) error {
	return json.Unmarshal(n.Data, dst)
}

func ParseChildNotification(data []byte) (*ChildNotification, error) {
	out := new(ChildNotification)
	return out, json.Unmarshal(data, out)
}

func NewChildNotificationStarted(info interface{}) ([]byte, error) {
	data, err := json.Marshal(info)
	if err != nil {
		return nil, fmt.Errorf("failed to marshal child info: %w", err)
	}

	return json.Marshal(ChildNotification{
		Kind: ChildNotificationKindStarted,
		Data: data,
	})
}

func NewChildNotificationError(err error) ([]byte, error) {
	data, err := json.Marshal(err.Error())
	if err != nil {
		return nil, fmt.Errorf("failed to marshal child error: %w", err)
	}

	return json.Marshal(ChildNotification{
		Kind: ChildNotificationKindError,
		Data: data,
	})
}
