package daemon

import (
	"errors"
	"io"
	"os"
)

var ErrCmdAlreadyBuilt = errors.New("command already built")

type Cmd struct {
	Executable      string
	Dir             string
	Args            []string
	Env             []string
	Stdout          *os.File
	Stderr          *os.File
	InfoPipe        *os.File
	built           bool
	childFiles      []*os.File
	closeAfterStart []io.Closer
	closeAfterWait  []io.Closer
}

func (c *Cmd) closeDescriptors(closers []io.Closer) {
	for _, fd := range closers {
		_ = fd.Close()
	}
}

func (c *Cmd) Close() {
	c.closeDescriptors(c.closeAfterStart)
	c.closeDescriptors(c.closeAfterWait)
}

func (c *Cmd) writerDescriptor(w *os.File) (f *os.File, err error) {
	if w == nil {
		f, err = os.OpenFile(os.DevNull, os.O_WRONLY, 0)
		if err != nil {
			return
		}
		c.closeAfterStart = append(c.closeAfterStart, f)
		return
	}

	return w, nil
}

func (c *Cmd) stdinFile() (*os.File, error) {
	f, err := os.Open(os.DevNull)
	if err != nil {
		return nil, err
	}

	c.closeAfterStart = append(c.closeAfterStart, f)
	return f, nil
}

func (c *Cmd) stdoutFile() (*os.File, error) {
	return c.writerDescriptor(c.Stdout)
}

func (c *Cmd) stderrFile() (*os.File, error) {
	if c.Stderr != nil && interfaceEqual(c.Stderr, c.Stdout) {
		return c.childFiles[1], nil
	}
	return c.writerDescriptor(c.Stderr)
}

func (c *Cmd) pipeFile() (*os.File, error) {
	pr, pw, err := os.Pipe()
	if err != nil {
		return nil, err
	}

	c.InfoPipe = pr
	c.closeAfterStart = append(c.closeAfterStart, pw)
	c.closeAfterWait = append(c.closeAfterWait, pr)
	return pw, nil
}

func interfaceEqual(a, b interface{}) bool {
	defer func() {
		_ = recover()
	}()
	return a == b
}
