package daemon

import (
	"context"
	"errors"
	"fmt"
	"io"
	"os"
	"strconv"
	"strings"
)

const (
	EnvChildMarkSuffix = "_GO_DAEMON"
	EnvParentPidSuffix = "_GO_DAEMON_PPID"
	EnvChildMarkValue  = "mlp"
)

var ErrChildDied = errors.New("child process unexpectedly died")

type Daemon struct {
	progname string
	dir      string
	env      []string
	args     []string
	stdout   *os.File
	stderr   *os.File
}

func NewDaemon(progName string, opts ...Option) *Daemon {
	d := &Daemon{
		progname: strings.ToUpper(progName),
		env:      os.Environ(),
		args:     os.Args,
	}

	for _, opt := range opts {
		opt(d)
	}

	return d
}

func (d *Daemon) StartChild(ctx context.Context) (*Child, error) {
	return d.startChild(ctx)
}

func (d *Daemon) NotifyStarted(info interface{}) error {
	data, err := NewChildNotificationStarted(info)
	if err != nil {
		return err
	}

	return d.sendToParent(data)
}

func (d *Daemon) NotifyError(err error) error {
	data, err := NewChildNotificationError(err)
	if err != nil {
		return err
	}

	return d.sendToParent(data)
}

func (d *Daemon) IsParent() bool {
	return os.Getenv(d.progname+EnvChildMarkSuffix) != EnvChildMarkValue
}

func (d *Daemon) Ppid() int {
	ppid := os.Getenv(d.progname + EnvParentPidSuffix)
	if ppid == "" {
		return 0
	}

	out, _ := strconv.Atoi(ppid)
	return out
}
func (d *Daemon) startChild(ctx context.Context) (*Child, error) {
	cmd, err := d.newCmd()
	if err != nil {
		return nil, fmt.Errorf("can't create cmd: %w", err)
	}
	defer cmd.Close()

	attr, err := cmd.BuildProcAttr()
	if err != nil {
		return nil, fmt.Errorf("can't build proc attrs: %w", err)
	}

	child, err := os.StartProcess(cmd.Executable, cmd.Args, attr)
	if err != nil {
		return nil, fmt.Errorf("unable to start process: %w", err)
	}

	cmd.closeDescriptors(cmd.closeAfterStart)

	infoChan := make(chan []byte)
	errChan := make(chan error)
	go func() {
		defer close(infoChan)
		defer close(errChan)

		infoBytes, err := io.ReadAll(cmd.InfoPipe)
		if err != nil {
			errChan <- fmt.Errorf("can't read info from child process: %w", err)
			return
		}
		infoChan <- infoBytes
	}()

	select {
	case infoBytes := <-infoChan:
		if len(infoBytes) == 0 {
			_ = child.Kill()
			return nil, ErrChildDied
		}

		cn, err := ParseChildNotification(infoBytes)
		if err != nil {
			_ = child.Kill()
			return nil, fmt.Errorf("child process sended invalid notification: %w", err)
		}

		if err := cn.Error(); err != nil {
			_ = child.Kill()
			return nil, fmt.Errorf("child reported startup error: %w", err)
		}

		defer func() {
			_ = child.Release()
		}()

		return &Child{
			pid:          child.Pid,
			notification: cn,
		}, nil
	case err := <-errChan:
		_ = child.Kill()
		return nil, err
	case <-ctx.Done():
		_ = child.Kill()
		return nil, ctx.Err()
	}
}

func (d *Daemon) newCmd() (*Cmd, error) {
	exe, err := os.Executable()
	if err != nil {
		return nil, fmt.Errorf("can't determine self exe: %w", err)
	}

	args := d.args
	if len(args) == 0 {
		args = []string{
			exe,
		}
	}

	out := Cmd{
		Executable: exe,
		Dir:        d.dir,
		Args:       args,
		Stdout:     d.stdout,
		Stderr:     d.stderr,
		Env: append(
			d.env,
			fmt.Sprintf("%s=%s", d.progname+EnvChildMarkSuffix, EnvChildMarkValue),
			fmt.Sprintf("%s=%d", d.progname+EnvParentPidSuffix, os.Getpid()),
		),
	}

	return &out, nil
}
