package gitconfig

import (
	"bufio"
	"errors"
	"fmt"
	"io"
	"strings"
)

type (
	parser struct {
		input *bufio.Reader
		eof   bool
	}

	Entries map[string]string
)

var (
	ErrEOF = errors.New("unexpected eof")
)

// Parse the given bytes as configuration file
func Parse(input io.Reader) (Entries, error) {
	ctx := &parser{
		input: bufio.NewReader(input),
		eof:   false,
	}

	return ctx.parse()
}

func (p *parser) parse() (Entries, error) {
	p.skipBom()

	comment := false
	result := map[string]string{}
	var name string
	var c byte
	var err error
	for {
		c, err = p.nextChar()
		if p.eof {
			// done!
			return result, nil
		}

		if err != nil {
			return nil, fmt.Errorf("failed to parse: %w", err)
		}

		if comment {
			if c == '\n' {
				comment = false
			}
			continue
		}

		switch c {
		case '\n':
			if p.eof {
				// done!
				return result, nil
			}
		case '\t', ' ':
			// skip
		case '#', ';':
			comment = true
		case '[':
			var nameBuilder strings.Builder
			err = p.parseSectionName(&nameBuilder)
			if err != nil {
				return nil, fmt.Errorf("failed to parse: %w", err)
			}
			nameBuilder.WriteByte('.')
			name = nameBuilder.String()
		default:
			if !isAlpha(c) {
				return nil, fmt.Errorf("failed to parse: unexpected char: %#v", c)
			}

			var nameBuilder strings.Builder
			nameBuilder.WriteString(name)
			nameBuilder.WriteByte(c | 0x20)

			value, err := p.parseEntry(&nameBuilder)
			if err != nil {
				return nil, fmt.Errorf("failed to parse: %w", err)
			}
			result[nameBuilder.String()] = value
		}
	}
}

func (p *parser) nextChar() (b byte, err error) {
	if p.eof {
		b = '\n'
		err = ErrEOF
		return
	}

	b, err = p.input.ReadByte()
	if err != nil {
		if err == io.EOF {
			b = '\n'
			err = ErrEOF
			p.eof = true
		}
		return
	}

	if b == '\r' {
		b, err = p.input.ReadByte()
		if err == nil && b == '\n' {
			return
		}

		err = nil
		_ = p.input.UnreadByte()
	}
	return
}

func (p *parser) parseSectionName(name *strings.Builder) error {
	var c byte
	var err error
	for {
		c, err = p.nextChar()
		if err != nil {
			return err
		}

		switch {
		case c == ']':
			return nil
		case isSpace(c):
			return p.parseSunSectionName(name)
		case !isKeyChar(c) && c != '.':
			return fmt.Errorf("unexpected char: %#v", c)
		default:
			name.WriteByte(c | 0x20)
		}
	}
}

// config: [BaseSection "ExtendedSection"]
func (p *parser) parseSunSectionName(name *strings.Builder) error {
	c, err := p.nextChar()
	for {
		if err != nil {
			return err
		}

		if !isSpace(c) {
			break
		}

		c, err = p.nextChar()
	}

	if c != '"' {
		return fmt.Errorf("expected quote but got: %#v", c)
	}

	name.WriteByte('.')
loop:
	for {
		c, err = p.nextChar()
		if err != nil {
			return err
		}

		switch c {
		case '\n':
			return fmt.Errorf("unexpected char: %#v", c)
		case '"':
			break loop
		case '\\':
			// skip it
		default:
			name.WriteByte(c | 0x20)
		}
	}

	c, err = p.nextChar()
	if err != nil {
		return err
	}

	if c != ']' {
		return fmt.Errorf("expected ] but got: %#v", c)
	}

	return nil
}

func (p *parser) parseEntry(name *strings.Builder) (string, error) {
	c, err := p.nextChar()
	for {
		if err != nil {
			return "", err
		}

		if !isKeyChar(c) {
			break
		}

		name.WriteByte(c | 0x20)
		c, err = p.nextChar()
	}

	var value string

	for c == ' ' || c == '\t' {
		c, err = p.nextChar()

		if err != nil {
			return "", err
		}
	}

	if c != '\n' {
		if c != '=' {
			return "", fmt.Errorf("expected = but got: %#v", c)
		}

		value, err = p.parseValue()
		if err != nil {
			return "", err
		}
	}

	return value, err
}

func (p *parser) parseValue() (string, error) {
	var c byte
	var err error
	var quote bool
	var value strings.Builder
	start := true
	for {
		c, err = p.nextChar()
		if p.eof {
			return value.String(), nil
		}

		if err != nil {
			return "", err
		}

		if start {
			if c == ' ' || c == '\t' {
				continue
			}
			start = false
		}
		switch c {
		case '"':
			quote = !quote
		case '\n':
			return value.String(), nil
		case '\\':
			c, err = p.nextChar()
			if err != nil {
				return "", err
			}

			switch c {
			case '\n':
				continue
			case 't':
				c = '\t'
			case 'b':
				c = '\b'
			case 'n':
				c = '\n'
			case '\\':
				break
			case '"':
				break
			default:
				return "", fmt.Errorf("unknown escape sequence: \\%v", c)
			}
			value.WriteByte(c)
		case ';', '#':
			if !quote {
				for c != '\n' {
					c, err = p.nextChar()
					if p.eof {
						// It's ok
						break
					}

					if err != nil {
						return "", err
					}
				}
				return value.String(), nil
			}
			value.WriteByte(c)
		default:
			value.WriteByte(c)
		}
	}
}

func (p *parser) skipBom() {
	bytes, err := p.input.Peek(3)
	if err != nil {
		// not enough bytes
		return
	}

	if bytes[0] == 0xef && bytes[1] == 0xbb && bytes[2] == 0xbf {
		_, _ = p.input.Discard(3)
	}
}

func isSpace(c byte) bool {
	return c == '\t' || c == ' ' || c == '\v' || c == '\f'
}

func isKeyChar(c byte) bool {
	return isAlpha(c) || (c >= '0' && c <= '9') || c == '-'
}

func isAlpha(c byte) bool {
	return c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z'
}
