package yrenderer

import (
	"net/url"

	"github.com/yuin/goldmark/ast"
	"github.com/yuin/goldmark/renderer"
	"github.com/yuin/goldmark/renderer/html"
	"github.com/yuin/goldmark/util"
)

var _ renderer.NodeRenderer = (*HideReferer)(nil)

type HideReferer struct {
	html.Config
}

func NewHideReferer(opts ...html.Option) renderer.NodeRenderer {
	r := &HideReferer{
		Config: html.NewConfig(),
	}
	for _, opt := range opts {
		opt.SetHTMLOption(&r.Config)
	}
	return r
}

// RegisterFuncs implements renderer.NodeRenderer.RegisterFuncs.
func (r *HideReferer) RegisterFuncs(reg renderer.NodeRendererFuncRegisterer) {
	reg.Register(ast.KindAutoLink, r.renderAutoLink)
	reg.Register(ast.KindLink, r.renderLink)
}

func (r *HideReferer) renderAutoLink(w util.BufWriter, source []byte, node ast.Node, entering bool) (ast.WalkStatus, error) {
	if !entering {
		return ast.WalkContinue, nil
	}

	n := node.(*ast.AutoLink)
	if n.AutoLinkType == ast.AutoLinkEmail {
		return ast.WalkContinue, nil
	}

	_, _ = w.WriteString(`<a href="https://h.yandex-team.ru/?`)
	uri := n.URL(source)
	label := n.Label(source)
	_, _ = w.Write(util.EscapeHTML([]byte(url.QueryEscape(string(uri)))))
	_, _ = w.WriteString(`" target="_blank"`)
	if n.Attributes() != nil {
		html.RenderAttributes(w, n, html.LinkAttributeFilter)
		_ = w.WriteByte('>')
	} else {
		_, _ = w.WriteString(`>`)
	}
	_, _ = w.Write(util.EscapeHTML(label))
	_, _ = w.WriteString(`</a>`)
	return ast.WalkContinue, nil
}

func (r *HideReferer) renderLink(w util.BufWriter, _ []byte, node ast.Node, entering bool) (ast.WalkStatus, error) {
	if !entering {
		_, _ = w.WriteString("</a>")
		return ast.WalkContinue, nil
	}

	n := node.(*ast.Link)
	_, _ = w.WriteString(`<a href="https://h.yandex-team.ru/?`)
	if r.Unsafe || !html.IsDangerousURL(n.Destination) {
		_, _ = w.Write(util.EscapeHTML([]byte(url.QueryEscape(string(n.Destination)))))
	}
	_, _ = w.WriteString(`" target="_blank"`)
	if n.Title != nil {
		_, _ = w.WriteString(` title="`)
		r.Writer.Write(w, n.Title)
		_ = w.WriteByte('"')
	}
	if n.Attributes() != nil {
		html.RenderAttributes(w, n, html.LinkAttributeFilter)
	}

	_ = w.WriteByte('>')
	return ast.WalkContinue, nil
}
