package yrenderer

import (
	"github.com/yuin/goldmark/ast"
	"github.com/yuin/goldmark/renderer"
	"github.com/yuin/goldmark/renderer/html"
	"github.com/yuin/goldmark/util"
)

var (
	_ renderer.NodeRenderer = (*MaxHeading)(nil)

	headings = []byte{
		'0',
		'1',
		'2',
		'3',
		'4',
		'5',
		'6',
	}
)

type MaxHeading struct {
	html.Config
	maxLevel int
}

func NewMaxHeading(maxLevel int, opts ...html.Option) renderer.NodeRenderer {
	r := &MaxHeading{
		Config:   html.NewConfig(),
		maxLevel: maxLevel,
	}

	for _, opt := range opts {
		opt.SetHTMLOption(&r.Config)
	}
	return r
}

// RegisterFuncs implements renderer.NodeRenderer.RegisterFuncs.
func (r *MaxHeading) RegisterFuncs(reg renderer.NodeRendererFuncRegisterer) {
	reg.Register(ast.KindHeading, r.renderHeading)
}

func (r *MaxHeading) renderHeading(w util.BufWriter, _ []byte, node ast.Node, entering bool) (ast.WalkStatus, error) {
	n := node.(*ast.Heading)
	lvl := n.Level
	if lvl > r.maxLevel {
		lvl = r.maxLevel
	}

	if entering {
		_, _ = w.WriteString("<h")
		_ = w.WriteByte(headings[lvl])
		if n.Attributes() != nil {
			html.RenderAttributes(w, node, html.HeadingAttributeFilter)
		}
		_ = w.WriteByte('>')
	} else {
		_, _ = w.WriteString("</h")
		_ = w.WriteByte(headings[lvl])
		_, _ = w.WriteString(">\n")
	}
	return ast.WalkContinue, nil
}
