package hashreader

import (
	"encoding/hex"
	"fmt"
	"hash"
	"io"
)

var _ io.Reader = (*HashedReader)(nil)

type HashedReader struct {
	hashName string
	hash     hash.Hash
	source   io.Reader
	size     int
}

func NewHashReader(in io.Reader, opts ...Option) (*HashedReader, error) {
	out := HashedReader{
		source: in,
	}

	defaultOpts := []Option{
		WithBlakeHash(),
	}
	for _, opt := range append(defaultOpts, opts...) {
		switch v := opt.(type) {
		case optionHash:
			out.hashName = v.name
			out.hash = v.hash
		}
	}

	return &out, nil
}

func (h *HashedReader) Read(p []byte) (int, error) {
	n, err := h.source.Read(p)
	if n > 0 {
		h.size += n
		_, _ = h.hash.Write(p[:n])
	}

	return n, err
}

func (h *HashedReader) Sum() string {
	return hex.EncodeToString(h.hash.Sum(nil))
}

func (h *HashedReader) Hash() string {
	return fmt.Sprintf("%s:%s", h.hashName, h.Sum())
}

func (h *HashedReader) Size() int {
	return h.size
}
