package ioatomic

import (
	"bytes"
	"io"
	"os"
	"path/filepath"
)

const (
	tmpSuffix = ".tmp-"
)

// WriteFileBytes file to temp and atomically move when everything else succeeds.
func WriteFileBytes(filename string, data []byte, perm os.FileMode) error {
	return WriteFile(filename, bytes.NewReader(data), perm)
}

// WriteFile file to temp and atomically move when everything else succeeds.
func WriteFile(filename string, data io.Reader, perm os.FileMode) error {
	dir, name := filepath.Split(filename)
	f, err := os.CreateTemp(dir, name+tmpSuffix)
	if err != nil {
		return err
	}

	defer func() { _ = f.Close() }()

	_, err = io.Copy(f, data)
	if err != nil {
		_ = os.Remove(f.Name())
		return err
	}

	err = os.Chmod(f.Name(), perm)
	if err != nil {
		_ = os.Remove(f.Name())
		return err
	}

	return os.Rename(f.Name(), filename)
}

// CopyFile file to temp and atomically move when everything else succeeds.
func CopyFile(from, to string) error {
	copyFile := func(from, to string) (string, error) {
		srcMode, err := os.Stat(from)
		if err != nil {
			return "", err
		}

		src, err := os.Open(from)
		if err != nil {
			return "", err
		}
		defer func() { _ = src.Close() }()

		dir, name := filepath.Split(to)
		dst, err := os.CreateTemp(dir, name+tmpSuffix)
		if err != nil {
			return "", err
		}
		defer func() { _ = dst.Close() }()

		_, err = io.Copy(dst, src)
		if err != nil {
			_ = os.Remove(dst.Name())
			return "", err
		}

		err = os.Chmod(dst.Name(), srcMode.Mode())
		if err != nil {
			_ = os.Remove(dst.Name())
			return "", err
		}

		return dst.Name(), nil
	}

	tmpPath, err := copyFile(from, to)
	if err != nil {
		return err
	}

	return os.Rename(tmpPath, to)
}
