package main

import (
	"bufio"
	"bytes"
	"flag"
	"fmt"
	"os"
	"regexp"
	"sort"
	"strconv"
	"strings"
	"text/template"

	"a.yandex-team.ru/library/go/yoimports/pkg/imports"
)

type ErrCode struct {
	CodeName string
	Code     string
	Desc     string
}

type TmplInfo struct {
	PackageName string
	ErrCodes    []ErrCode
}

const errCodesTmpl = `
// Code generated by iokit-errgen. DO NOT EDIT.
package {{.PackageName}}

var (
{{range .ErrCodes -}}
Err{{.CodeName}} = &Error{RetCode: {{.Code}}, Msg: {{.Desc}}}
{{end -}}
)

func iokitErrorFromCode(rc int64) *Error {
	switch rc {
{{range .ErrCodes -}}
		case {{.Code}}:
			return Err{{.CodeName}}
{{end -}}
		default:
			return &Error{RetCode: rc, Msg: "unknown iokit return code"}
	}
}
`

func fatalf(msg string, args ...interface{}) {
	line := fmt.Sprintf(msg, args...)
	_, _ = os.Stderr.WriteString(line + "\n")
	os.Exit(2)
}

func main() {
	var inFilePath, outFilePath, pkgName string
	flag.StringVar(&inFilePath, "in", "iokit_errors", "input filename")
	flag.StringVar(&outFilePath, "out", "iokit_errors_codes.go", "output filename")
	flag.StringVar(&pkgName, "pkg-name", "iokit", "output package name")
	flag.Parse()

	inFile, err := os.Open(inFilePath)
	if err != nil {
		fatalf("can't open input file: %v", err)
	}
	defer func() { _ = inFile.Close() }()

	reError := regexp.MustCompile(`^#define\s+kIOReturn(?P<CodeName>[a-zA-Z0-9]+)\s+iokit_common_err\(0x(?P<Code>[a-fA-F0-9]+)\)\s*//\s*(?P<Desc>.*)\s*$`)
	scanner := bufio.NewScanner(inFile)
	var errCodes []ErrCode
	for scanner.Scan() {
		l := scanner.Text()
		res := reError.FindStringSubmatch(l)
		if len(res) == 0 {
			continue
		}

		names := reError.SubexpNames()
		var errCode ErrCode
		for i, _ := range res {
			switch names[i] {
			case "CodeName":
				errCode.CodeName = strings.TrimSpace(res[i])
			case "Code":
				rawCode, err := strconv.ParseInt(strings.TrimSpace(res[i]), 16, 64)
				if err != nil {
					fatalf("unable to parse err code: %s", res[i])
				}

				errCode.Code = fmt.Sprintf("0x%08x", 0xe0000000+rawCode)
			case "Desc":
				desc := strings.TrimSpace(res[i])
				desc = strings.ToLower(string(desc[0])) + desc[1:]

				errCode.Desc = strconv.Quote(desc)
			}
		}

		if errCode.Code == "" || errCode.CodeName == "" {
			fatalf("invalid errcode: %s", l)
		}

		errCodes = append(errCodes, errCode)
	}

	if err := scanner.Err(); err != nil {
		fatalf("failed to iterate over input file: %v", err)
	}

	sort.Slice(errCodes, func(i, j int) bool {
		return errCodes[i].Code < errCodes[j].Code
	})

	tmplInfo := TmplInfo{
		PackageName: pkgName,
		ErrCodes:    errCodes,
	}

	var out bytes.Buffer
	err = template.Must(template.New("errs").Parse(errCodesTmpl[1:])).Execute(&out, tmplInfo)
	if err != nil {
		fatalf("can't generate err codes: %v", err)
	}

	formatted, err := imports.Process(out.Bytes())
	if err != nil {
		// Print out the bad code with line numbers.
		// This should never happen in practice, but it can while changing generated code,
		// so consider this a debugging aid.
		var src bytes.Buffer
		s := bufio.NewScanner(bytes.NewReader(out.Bytes()))
		for line := 1; s.Scan(); line++ {
			_, _ = fmt.Fprintf(&src, "%5d\t%s\n", line, s.Bytes())
		}

		fatalf("bad Go source code was generated: %v\n%s", err, src.String())
	}

	err = os.WriteFile(outFilePath, formatted, 0o644)
	if err != nil {
		fatalf("can't write out file: %v", err)
	}
}
