#include "iokit_darwin.h"

static kern_return_t fillDeviceInfo(io_service_t device, struct Device* out);
static kern_return_t replugDevice(io_service_t device);
static kern_return_t reEnumerateDevice(IOUSBDeviceInterface187** deviceInterface);

kern_return_t List(uint32_t vendorID, uint32_t productID, struct Device out[]) {
    CFMutableDictionaryRef 	matchingDict = NULL;
	CFNumberRef				numberRef = NULL;
	kern_return_t			kr = 0;
	io_iterator_t			foundDevices = 0;
	io_object_t				device = 0;
    int                     i;

	matchingDict = IOServiceMatching(kIOUSBDeviceClassName);
	if (matchingDict == NULL) {
		return kIOReturnError;
	}

	numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &vendorID);
	if (numberRef == NULL) {
	    return kIOReturnNoMemory;
	}
	CFDictionarySetValue(matchingDict, CFSTR(kUSBVendorID), numberRef);
	CFRelease(numberRef);

	if (productID > 0) {
		numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &productID);
        if (numberRef == NULL) {
            return kIOReturnNoMemory;
        }
		CFDictionarySetValue(matchingDict, CFSTR(kUSBProductID),  numberRef);
		CFRelease(numberRef);
	} else {
		CFDictionarySetValue(matchingDict, CFSTR(kUSBProductID),  CFSTR(kAnyProductId));
	}

    kr = IOServiceGetMatchingServices(kIOMasterPortDefault, matchingDict, &foundDevices);
	// IOServiceGetMatchingServices consumes matchingDict reference
    matchingDict = NULL;
    if (kr) {
        return kr;
    }

    i = 0;
    kr = 0;
	while (i < NDEVICE && (device = IOIteratorNext(foundDevices))) {
		kr = fillDeviceInfo(device, &out[i]);
		IOObjectRelease(device);

		if (kr) {
		    break;
		}

        i++;
	}

	IOObjectRelease(foundDevices);
	return kr;
}

kern_return_t Replug(uint32_t locationID) {
    CFMutableDictionaryRef 	matchingDict = NULL;
    CFMutableDictionaryRef  propertyMatchingDict = NULL;
	CFNumberRef				locationIDRef = NULL;
	kern_return_t			kr = 0;
	io_iterator_t			foundDevices = 0;
	io_object_t				device = 0;

	matchingDict = IOServiceMatching(kIOUSBDeviceClassName);
	if (matchingDict == NULL) {
		return kIOReturnError;
	}

    propertyMatchingDict = CFDictionaryCreateMutable(kCFAllocatorDefault, 0, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
	if (propertyMatchingDict == NULL) {
	    return kIOReturnNoMemory;
	}

    // Set the value in the dictionary of the property with the given key, or add the key
    // to the dictionary if it doesn't exist. This call retains the value object passed in.
	locationIDRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &locationID);
    if (locationIDRef == NULL) {
        return kIOReturnNoMemory;
    }
    CFDictionarySetValue(propertyMatchingDict, CFSTR(kLocationID), locationIDRef);
    CFRelease(locationIDRef);

    // Now add the dictionary containing the matching value to our main
    // matching dictionary. This call will retain propertyMatchDict,
    CFDictionarySetValue(matchingDict, CFSTR(kIOPropertyMatchKey), propertyMatchingDict);

    kr = IOServiceGetMatchingServices(kIOMasterPortDefault, matchingDict, &foundDevices);
	// IOServiceGetMatchingServices consumes matchingDict reference
    matchingDict = NULL;
    if (kr) {
        return kr;
    }

    device = IOIteratorNext(foundDevices);
    if (!device) {
        return kIOReturnNotFound;
    }

    kr = replugDevice(device);
    IOObjectRelease(device);
    IOObjectRelease(foundDevices);
    return kr;
}

static kern_return_t fillDeviceInfo(io_service_t device, struct Device* out) {
    CFNumberRef	numberRef = NULL;
    IOReturn	kr;

    memset(out, 0, sizeof *out);

    kr = IORegistryEntryGetName(device, out->name);
    if (kr) {
        return kr;
    }

	numberRef = IORegistryEntryCreateCFProperty(device, CFSTR(kLocationID), kCFAllocatorDefault, 0);
	if (numberRef == NULL) {
		return kIOReturnNotFound;
	}

    CFNumberGetValue(numberRef, kCFNumberSInt32Type, &out->locationID);
    CFRelease(numberRef);
	return kIOReturnSuccess;
}

static kern_return_t reEnumerateDevice(IOUSBDeviceInterface187** deviceInterface) {
	IOReturn kr = 0;
	kr = (*deviceInterface)->USBDeviceOpen(deviceInterface);
	if (kr) {
		return kr;
	}

	kr = (*deviceInterface)->USBDeviceReEnumerate(deviceInterface, 0);

	// cleanup
	(void) (*deviceInterface)->USBDeviceClose(deviceInterface);
	(*deviceInterface)->Release(deviceInterface);

	return kr;
}

static kern_return_t replugDevice(io_service_t device) {
	IOCFPlugInInterface**		plugInInterface = NULL;
	IOUSBDeviceInterface187**	deviceInterface = NULL;
	SInt32						score = 0;
	HRESULT						res = NULL;
	IOReturn					kr = 0;

	kr = IOCreatePlugInInterfaceForService(device, kIOUSBDeviceUserClientTypeID, kIOCFPlugInInterfaceID, &plugInInterface, &score);
	IOObjectRelease(device);
	if (kr) {
		return kr;
	}

	if (plugInInterface == NULL) {
		return kIOReturnNotFound;
	}

	// Use the plugin interface to retrieve the device interface.
	res = (*plugInInterface)->QueryInterface(plugInInterface, CFUUIDGetUUIDBytes(kIOUSBDeviceInterfaceID187), (LPVOID *)&deviceInterface);
	// Now done with the plugin interface.
	(*plugInInterface)->Release(plugInInterface);

	if (res || deviceInterface == NULL) {
    	return kIOReturnNotFound;
  	}

	return reEnumerateDevice(deviceInterface);
}
