package lineage

import (
	"bufio"
	"bytes"
	"errors"
	"os"
	"path/filepath"
	"strconv"
	"strings"
)

type OS struct {
	Family   string
	Release  string
	Codename string
}

func CurrentOS() (*OS, error) {
	return OSInRoot("/")
}

func OSInRoot(root string) (*OS, error) {
	if detectedOS := ParseLsbRelease(filepath.Join(root, LsbReleasePath)); detectedOS != nil {
		return detectedOS, nil
	}

	if detectedOS := ParseOsRelease(filepath.Join(root, OsReleasePath)); detectedOS != nil {
		return detectedOS, nil
	}

	if detectedOS := ParseDebianVersion(filepath.Join(root, DebianVersionPath)); detectedOS != nil {
		return detectedOS, nil
	}

	if detectedOS := ParseAlpineRelease(filepath.Join(root, AlpineReleasePath)); detectedOS != nil {
		return detectedOS, nil
	}

	return nil, errors.New("unsupported OS")
}

func ParseLsbRelease(path string) (result *OS) {
	lsbRelease, err := os.Open(path)
	if err != nil {
		return
	}
	defer func() { _ = lsbRelease.Close() }()

	result = new(OS)
	scanner := bufio.NewScanner(lsbRelease)
	for scanner.Scan() {
		line := scanner.Text()
		switch {
		case strings.HasPrefix(line, "DISTRIB_ID="):
			dID := strings.TrimSpace(line[11:])
			if unquoted, err := strconv.Unquote(dID); err == nil {
				dID = unquoted
			}

			switch dID {
			case "ManjaroLinux":
				result.Family = Manjaro
			case "Arch":
				result.Family = ArchLinux
			case "Ubuntu":
				result.Family = Ubuntu
			default:
				result.Family = dID
			}
		case strings.HasPrefix(line, "DISTRIB_RELEASE="):
			result.Release = strings.TrimSpace(line[16:])
			if unquoted, err := strconv.Unquote(result.Release); err == nil {
				result.Release = unquoted
			}
		case strings.HasPrefix(line, "DISTRIB_CODENAME="):
			result.Codename = strings.TrimSpace(line[17:])
			if unquoted, err := strconv.Unquote(result.Codename); err == nil {
				result.Codename = unquoted
			}
		}
	}

	if result.Family == "" {
		return nil
	}
	return fixup(result)
}

// https://gist.github.com/natefoo/814c5bf936922dad97ff
// https://www.freedesktop.org/software/systemd/man/os-release.html
func ParseOsRelease(path string) (result *OS) {
	osRelease, err := os.Open(path)
	if err != nil {
		return
	}
	defer func() { _ = osRelease.Close() }()

	result = new(OS)
	scanner := bufio.NewScanner(osRelease)
	for scanner.Scan() {
		line := scanner.Text()
		switch {
		case strings.HasPrefix(line, "ID="):
			osID := strings.TrimSpace(line[3:])
			if unquoted, err := strconv.Unquote(osID); err == nil {
				osID = unquoted
			}
			switch osID {
			case "alpine":
				result.Family = Alpine
			case "manjaro":
				result.Family = Manjaro
			case "arch":
				result.Family = ArchLinux
			case "ubuntu":
				result.Family = Ubuntu
			case "slackware":
				result.Family = Slackware
			case "opensuse":
				result.Family = OpenSUSE
			case "fedora":
				result.Family = Fedora
			case "debian":
				result.Family = Debian
			case "centos":
				result.Family = CentOS
			default:
				result.Family = osID
			}
		case strings.HasPrefix(line, "VERSION_ID="):
			result.Release = strings.TrimSpace(line[11:])
			if unquoted, err := strconv.Unquote(result.Release); err == nil {
				result.Release = unquoted
			}
		case strings.HasPrefix(line, "VERSION_CODENAME="):
			result.Codename = strings.TrimSpace(line[17:])
			if unquoted, err := strconv.Unquote(result.Codename); err == nil {
				result.Codename = unquoted
			}
		}
	}

	if result.Family == "" {
		return nil
	}

	return fixup(result)
}

func ParseDebianVersion(path string) *OS {
	rawVersion, err := os.ReadFile(path)
	if err != nil {
		return nil
	}

	result := &OS{
		Family:  Debian,
		Release: string(bytes.TrimSpace(rawVersion)),
	}
	return fixup(result)
}

func ParseAlpineRelease(path string) *OS {
	alpineVersion, err := os.ReadFile(path)
	if err != nil {
		return nil
	}

	result := &OS{
		Family:  Alpine,
		Release: string(bytes.TrimSpace(alpineVersion)),
	}
	return fixup(result)
}

func fixup(info *OS) *OS {
	if info == nil {
		return nil
	}

	if info.Codename != "" {
		return info
	}

	if info.Release == "" {
		return info
	}

	switch info.Family {
	case Debian:
		info.Codename = debianCodeName(info.Release)
	case Ubuntu:
		info.Codename = ubuntuCodeName(info.Release)
	case Alpine:
		info.Codename = alpineCodeName(info.Release)
	}
	return info
}

// Reference: https://www.debian.org/releases/
func debianCodeName(version string) string {
	if strings.HasSuffix(version, "/sid") {
		return "sid"
	}

	if idx := strings.Index(version, "."); idx > 0 {
		version = version[:idx]
	}

	switch version {
	case "13":
		return "trixie"
	case "12":
		return "bookworm"
	case "11":
		return "bullseye"
	case "10":
		return "buster"
	case "9":
		return "stretch"
	case "8":
		return "jessie"
	case "7":
		return "wheezy"
	case "6":
		return "squeeze"
	case "5":
		return "lenny"
	case "4":
		return "etch"
	case "":
		// sid version doesn't have version_id
		return "sid"
	default:
		return ""
	}
}

// Reference: https://wiki.ubuntu.com/Releases
func ubuntuCodeName(version string) string {
	parts := strings.SplitN(version, ".", 3)
	if len(parts) > 2 {
		version = strings.Join(parts[:2], ".")
	}

	switch version {
	case "22.04":
		return "jammy"
	case "21.10":
		return "impish"
	case "21.04":
		return "hirsute"
	case "20.10":
		return "groovy"
	case "20.04":
		return "focal"
	case "19.10":
		return "eoan"
	case "19.04":
		return "disco"
	case "18.10":
		return "cosmic"
	case "18.04":
		return "bionic"
	case "17.10":
		return "artful"
	case "17.04":
		return "zesty"
	case "16.10":
		return "yakkety"
	case "16.04":
		return "xenial"
	case "15.10":
		return "wily"
	case "15.04":
		return "vivid"
	case "14.10":
		return "utopic"
	case "14.04":
		return "trusty"
	case "13.10":
		return "saucy"
	case "13.04":
		return "raring"
	case "12.10":
		return "quantal"
	case "12.04":
		return "precise"
	case "11.10":
		return "oneiric"
	case "11.04":
		return "natty"
	case "10.10":
		return "maverick"
	case "10.04":
		return "lucid"
	default:
		return ""
	}
}

func alpineCodeName(version string) string {
	parts := strings.SplitN(version, ".", 3)
	if len(parts) > 2 {
		version = strings.Join(parts[:2], ".")
	}

	return "v" + version
}
