package main

import (
	"bufio"
	"bytes"
	"flag"
	"fmt"
	"os"
	"regexp"
	"sort"
	"strconv"
	"strings"
	"text/template"

	"github.com/iancoleman/strcase"

	"a.yandex-team.ru/library/go/yoimports/pkg/imports"
)

type ErrCode struct {
	CodeName string
	Code     string
	Desc     string
}

type TmplInfo struct {
	PackageName string
	ErrCodes    []ErrCode
}

const errCodesTmpl = `
// Code generated by pcsc-errgen. DO NOT EDIT.
package {{.PackageName}}

import "fmt"

var (
{{range .ErrCodes -}}
Err{{.CodeName}} = &Error{RetCode: {{.Code}}, Msg: {{.Desc}}}
{{end -}}
)

func pcscErrorFromCode(rc int64) *Error {
	switch rc {
{{range .ErrCodes -}}
		case {{.Code}}:
			return Err{{.CodeName}}
{{end -}}
		default:
			return &Error{RetCode: rc, Msg: fmt.Sprintf("unknown pcsc return code 0x%08x", rc)}
	}
}
`

func fatalf(msg string, args ...interface{}) {
	line := fmt.Sprintf(msg, args...)
	_, _ = os.Stderr.WriteString(line + "\n")
	os.Exit(2)
}

func main() {
	var inFilePath, outFilePath, pkgName string
	flag.StringVar(&inFilePath, "in", "pcsc_errors", "input filename")
	flag.StringVar(&outFilePath, "out", "pcsc_errors_codes.go", "output filename")
	flag.StringVar(&pkgName, "pkg-name", "pcsc", "output package name")
	flag.Parse()

	inFile, err := os.Open(inFilePath)
	if err != nil {
		fatalf("can't open input file: %v", err)
	}
	defer func() { _ = inFile.Close() }()

	reError := regexp.MustCompile(`^#define\s+SCARD_[A-Z_]_(?P<CodeName>[A-Z_]+)\s+\((?:\(LONG\))?(?P<Code>0x[a-fA-F0-9]+)\)\s*/\*\*<\s*(?P<Desc>.*)\.\s*\*/$`)
	scanner := bufio.NewScanner(inFile)
	var errCodes []ErrCode
	for scanner.Scan() {
		l := scanner.Text()
		res := reError.FindStringSubmatch(l)
		if len(res) == 0 {
			continue
		}

		names := reError.SubexpNames()
		var errCode ErrCode
		for i, _ := range res {
			switch names[i] {
			case "CodeName":
				errCode.CodeName = strcase.ToCamel(strings.ToLower(strings.TrimSpace(res[i])))
			case "Code":
				errCode.Code = strings.TrimSpace(res[i])
			case "Desc":
				desc := strings.TrimSpace(res[i])
				desc = strings.ToLower(string(desc[0])) + desc[1:]

				errCode.Desc = strconv.Quote(desc)
			}
		}

		if errCode.Code == "" || errCode.CodeName == "" {
			fatalf("invalid errcode: %s", l)
		}

		errCodes = append(errCodes, errCode)
	}

	if err := scanner.Err(); err != nil {
		fatalf("failed to iterate over input file: %v", err)
	}

	sort.Slice(errCodes, func(i, j int) bool {
		return errCodes[i].Code < errCodes[j].Code
	})

	tmplInfo := TmplInfo{
		PackageName: pkgName,
		ErrCodes:    errCodes,
	}

	var out bytes.Buffer
	err = template.Must(template.New("errs").Parse(errCodesTmpl[1:])).Execute(&out, tmplInfo)
	if err != nil {
		fatalf("can't generate err codes: %v", err)
	}

	formatted, err := imports.Process(out.Bytes())
	if err != nil {
		// Print out the bad code with line numbers.
		// This should never happen in practice, but it can while changing generated code,
		// so consider this a debugging aid.
		var src bytes.Buffer
		s := bufio.NewScanner(bytes.NewReader(out.Bytes()))
		for line := 1; s.Scan(); line++ {
			_, _ = fmt.Fprintf(&src, "%5d\t%s\n", line, s.Bytes())
		}

		fatalf("bad Go source code was generated: %v\n%s", err, src.String())
	}

	err = os.WriteFile(outFilePath, formatted, 0o644)
	if err != nil {
		fatalf("can't write out file: %v", err)
	}
}
