package pcsc

import "C"

// https://ludovicrousseau.blogspot.com/2010/04/pcsc-sample-in-c.html

// #cgo darwin LDFLAGS: -framework PCSC
// #include <PCSC/winscard.h>  // Y_IGNORE
// #include <PCSC/wintypes.h>  // Y_IGNORE
import "C"

import (
	"bytes"
	"fmt"
	"unsafe"
)

const rcSuccess = C.SCARD_S_SUCCESS

var _ Client = (*scContext)(nil)

type scContext struct {
	ctx C.SCARDCONTEXT
}

func NewClient() (Client, error) {
	var ctx C.SCARDCONTEXT
	rc := C.SCardEstablishContext(C.SCARD_SCOPE_SYSTEM, nil, nil, &ctx)
	if err := scCheck(rc); err != nil {
		return nil, err
	}
	return &scContext{ctx: ctx}, nil
}

func (c *scContext) CheckCompatibility() error {
	return nil
}

func (c *scContext) ListReaders() ([]string, error) {
	var n C.DWORD
	rc := C.SCardListReaders(c.ctx, nil, nil, &n)
	if err := scCheck(rc); err != nil {
		return nil, err
	}

	d := make([]byte, n)
	rc = C.SCardListReaders(c.ctx, nil, (*C.char)(unsafe.Pointer(&d[0])), &n)
	if err := scCheck(rc); err != nil {
		return nil, err
	}

	var readers []string
	for _, d := range bytes.Split(d, []byte{0}) {
		if len(d) == 0 {
			continue
		}

		readers = append(readers, string(d))
	}
	return readers, nil
}

func (c *scContext) Connect(reader string) (Handle, error) {
	var (
		handle         C.SCARDHANDLE
		activeProtocol C.DWORD
	)
	rc := C.SCardConnect(c.ctx, C.CString(reader),
		C.SCARD_SHARE_SHARED, C.SCARD_PROTOCOL_T1,
		&handle, &activeProtocol)
	if err := scCheck(rc); err != nil {
		return nil, err
	}
	return &scHandle{handle}, nil
}

func (c *scContext) Close() error {
	return scCheck(C.SCardReleaseContext(c.ctx))
}

var _ Handle = (*scHandle)(nil)

type scHandle struct {
	h C.SCARDHANDLE
}

func (h *scHandle) Begin() (Tx, error) {
	if err := scCheck(C.SCardBeginTransaction(h.h)); err != nil {
		return nil, err
	}
	return &scTx{h.h}, nil
}

func (h *scHandle) Close() error {
	return scCheck(C.SCardDisconnect(h.h, C.SCARD_LEAVE_CARD))
}

var _ Tx = (*scTx)(nil)

type scTx struct {
	h C.SCARDHANDLE
}

func (t *scTx) Close() error {
	return scCheck(C.SCardEndTransaction(t.h, C.SCARD_LEAVE_CARD))
}

func (t *scTx) transmit(req []byte) (more bool, b []byte, err error) {
	var resp [C.MAX_BUFFER_SIZE_EXTENDED]byte
	reqN := C.DWORD(len(req))
	respN := C.DWORD(len(resp))
	rc := C.SCardTransmit(
		t.h,
		C.SCARD_PCI_T1,
		(*C.BYTE)(&req[0]), reqN, nil,
		(*C.BYTE)(&resp[0]), &respN)
	if err := scCheck(rc); err != nil {
		return false, nil, fmt.Errorf("transmitting request: %w", err)
	}
	if respN < 2 {
		return false, nil, fmt.Errorf("scard response too short: %d", respN)
	}
	sw1 := resp[respN-2]
	sw2 := resp[respN-1]
	if sw1 == 0x90 && sw2 == 0x00 {
		return false, resp[:respN-2], nil
	}
	if sw1 == 0x61 {
		return true, resp[:respN-2], nil
	}
	return false, nil, newError(sw1, sw2)
}

func scCheck(rc C.int) error {
	if rc == rcSuccess {
		return nil
	}

	i := int64(rc)
	if i < 0 {
		// On MacOS, int isn't big enough to handle the return codes so the
		// leading bit becomes a two's complement bit. If the return code is
		// negative, correct this.
		// https://github.com/go-piv/piv-go/issues/53
		i += 1 << 32
	}
	return pcscErrorFromCode(i)
}
