package piv

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/des"
	"fmt"
)

type ManagementKeyAlgo uint8

const (
	ManagementKeyAlgoNone   ManagementKeyAlgo = 0
	ManagementKeyAlgoTDES   ManagementKeyAlgo = 0x03
	ManagementKeyAlgoAES128 ManagementKeyAlgo = 0x08
	ManagementKeyAlgoAES192 ManagementKeyAlgo = 0x0A
	ManagementKeyAlgoAES256 ManagementKeyAlgo = 0x0C
)

func (t ManagementKeyAlgo) String() string {
	switch t {
	case ManagementKeyAlgoTDES:
		return "TDES"
	case ManagementKeyAlgoAES128:
		return "AES128"
	case ManagementKeyAlgoAES192:
		return "AES192"
	case ManagementKeyAlgoAES256:
		return "AES256"
	default:
		return fmt.Sprintf("unknown_%x", t.Byte())
	}
}

func (t ManagementKeyAlgo) KeyLen() int {
	switch t {
	case ManagementKeyAlgoTDES:
		return 24
	case ManagementKeyAlgoAES128:
		return 16
	case ManagementKeyAlgoAES192:
		return 24
	case ManagementKeyAlgoAES256:
		return 32
	default:
		return 0
	}
}

func (t ManagementKeyAlgo) ChallengeLen() int {
	switch t {
	case ManagementKeyAlgoTDES:
		return 8
	default:
		return 16
	}
}

func (t ManagementKeyAlgo) Byte() byte {
	return byte(t)
}

func ParseManagementKeyType(in []byte) (ManagementKeyAlgo, error) {
	if len(in) != 1 {
		return ManagementKeyAlgoNone, fmt.Errorf("unexpected input bytes len: %d", len(in))
	}

	out := ManagementKeyAlgo(in[0])
	switch out {
	case ManagementKeyAlgoTDES, ManagementKeyAlgoAES128, ManagementKeyAlgoAES192, ManagementKeyAlgoAES256:
		return out, nil
	default:
		return ManagementKeyAlgoNone, fmt.Errorf("unsupported key algo: %x", in)
	}
}

type ManagementKeyMetadata struct {
	Algo        ManagementKeyAlgo
	TouchPolicy TouchPolicy
}

type ManagementKey struct {
	ManagementKeyMetadata
	Key []byte
}

func (m *ManagementKey) Cipher() (cipher.Block, error) {
	switch m.Algo {
	case ManagementKeyAlgoTDES:
		return des.NewTripleDESCipher(m.Key)
	case ManagementKeyAlgoAES128, ManagementKeyAlgoAES192, ManagementKeyAlgoAES256:
		return aes.NewCipher(m.Key)
	default:
		return nil, fmt.Errorf("unsupported key algo: %s", m.Algo)
	}
}
