package piv

import (
	"bytes"
	"crypto/rand"
	"encoding/asn1"
	"encoding/binary"
	"errors"
	"fmt"
	"io"
	"math/big"

	"a.yandex-team.ru/security/libs/go/pcsc"
)

var (
	// DefaultPIN for the PIV applet. The PIN is used to change the Management Key,
	// and slots can optionally require it to perform signing operations.
	DefaultPIN = "123456"
	// DefaultPUK for the PIV applet. The PUK is only used to reset the PIN when
	// the card's PIN retries have been exhausted.
	DefaultPUK = "12345678"
	// DefaultManagementKey for the PIV applet. The Management Key is a Triple-DES
	// key required for slot actions such as generating keys, setting certificates,
	// and signing.
	DefaultManagementKey = ManagementKey{
		ManagementKeyMetadata: ManagementKeyMetadata{
			Algo:        ManagementKeyAlgoTDES,
			TouchPolicy: TouchPolicyNever,
		},
		Key: []byte{
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
		},
	}
)

// Cards lists all smart cards available via PC/SC interface. Card names are
// strings describing the key, such as "Yubico Yubikey NEO OTP+U2F+CCID 00 00".
//
// Card names depend on the operating system and what port a card is plugged
// into. To uniquely identify a card, use its serial number.
//
// See: https://ludovicrousseau.blogspot.com/2010/05/what-is-in-pcsc-reader-name.html
func Cards() ([]string, error) {
	var c client
	return c.Cards()
}

const (
	// https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-78-4.pdf#page=17
	algTag     = 0x80
	alg3DES    = 0x03
	algRSA1024 = 0x06
	algRSA2048 = 0x07
	algECCP256 = 0x11
	algECCP384 = 0x14
	// non-standard; as implemented by SoloKeys. Chosen for low probability of eventual
	// clashes, if and when PIV standard adds Ed25519 support
	algEd25519 = 0x22

	// https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-78-4.pdf#page=16
	keyAuthentication     = 0x9a
	keyCardManagement     = 0x9b
	keySignature          = 0x9c
	keyKeyManagement      = 0x9d
	keyCardAuthentication = 0x9e
	keyAttestation        = 0xf9

	insVerify             = 0x20
	insChangeReference    = 0x24
	insResetRetry         = 0x2c
	insGenerateAsymmetric = 0x47
	insAuthenticate       = 0x87
	insGetData            = 0xcb
	insPutData            = 0xdb
	insSelectApplication  = 0xa4
	insGetResponseAPDU    = 0xc0

	// https://github.com/Yubico/yubico-piv-tool/blob/4f6e91af63d53a97a25bc3aa350160891309d960/lib/ykpiv.h
	insSetMGMKey     = 0xff
	insImportKey     = 0xfe
	insGetVersion    = 0xfd
	insReset         = 0xfb
	insSetPINRetries = 0xfa
	insAttest        = 0xf9
	insGetSerial     = 0xf8
	insGetMetadata   = 0xf7

	// https://github.com/Yubico/yubico-piv-tool/blob/4f6e91af63d53a97a25bc3aa350160891309d960/lib/ykpiv.h#L724-L730
	metadataAlgoTag   = 0x01
	metadataPolicyTag = 0x02
)

// YubiKey is an exclusive open connection to a YubiKey smart card. While open,
// no other process can query the given card.
//
// To release the connection, call the Close method.
type YubiKey struct {
	scClient pcsc.Client
	scHandle pcsc.Handle
	scTx     pcsc.Tx

	rand io.Reader

	// Used to determine how to access certain functionality.
	//
	// TODO: It's not clear what this actually communicates. Is this the
	// YubiKey's version or PIV version? A NEO reports v1.0.4. Figure this out
	// before exposing an API.
	version *version
}

// Close releases the connection to the smart card.
func (yk *YubiKey) Close() error {
	err1 := yk.scHandle.Close()
	err2 := yk.scClient.Close()
	if err1 == nil {
		return err2
	}
	return err1
}

// Open connects to a YubiKey smart card.
func Open(card string) (*YubiKey, error) {
	var c client
	return c.Open(card)
}

// client is a smart card client and may be exported in the future to allow
// configuration for the top level Open() and Cards() APIs.
type client struct {
	// Rand is a cryptographic source of randomness used for card challenges.
	//
	// If nil, defaults to crypto.Rand.
	Rand io.Reader
}

func (c *client) Cards() ([]string, error) {
	pc, err := pcsc.NewClient()
	if err != nil {
		return nil, fmt.Errorf("connecting to pscs: %w", err)
	}
	defer func() { _ = pc.Close() }()

	return pc.ListReaders()
}

func (c *client) Open(card string) (*YubiKey, error) {
	pc, err := pcsc.NewClient()
	if err != nil {
		return nil, fmt.Errorf("connecting to smart card daemon: %w", err)
	}

	h, err := pc.Connect(card)
	if err != nil {
		_ = pc.Close()
		return nil, fmt.Errorf("connecting to smart card: %w", err)
	}

	tx, err := h.Begin()
	if err != nil {
		return nil, fmt.Errorf("beginning smart card transaction: %w", err)
	}

	if err := ykSelectApplication(tx, aidPIV[:]); err != nil {
		_ = tx.Close()
		return nil, fmt.Errorf("selecting piv applet: %w", err)
	}

	yk := &YubiKey{scClient: pc, scHandle: h, scTx: tx}
	v, err := ykVersion(yk.scTx)
	if err != nil {
		_ = yk.Close()
		return nil, fmt.Errorf("getting yubikey version: %w", err)
	}
	yk.version = v
	if c.Rand != nil {
		yk.rand = c.Rand
	} else {
		yk.rand = rand.Reader
	}
	return yk, nil
}

// Version returns the version as reported by the PIV applet. For newer
// YubiKeys (>=4.0.0) this corresponds to the version of the YubiKey itself.
//
// Older YubiKeys return values that aren't directly related to the YubiKey
// version. For example, 3rd generation YubiKeys report 1.0.X.
func (yk *YubiKey) Version() Version {
	return Version{
		Major: int(yk.version.major),
		Minor: int(yk.version.minor),
		Patch: int(yk.version.patch),
	}
}

// Serial returns the YubiKey's serial number.
func (yk *YubiKey) Serial() (uint32, error) {
	return ykSerial(yk.scTx, yk.version)
}

func encodePIN(pin string) ([]byte, error) {
	data := []byte(pin)
	if len(data) == 0 {
		return nil, fmt.Errorf("pin cannot be empty")
	}
	if len(data) > 8 {
		return nil, fmt.Errorf("pin longer than 8 bytes")
	}
	// apply padding
	for i := len(data); i < 8; i++ {
		data = append(data, 0xff)
	}
	return data, nil
}

// authPIN attempts to authenticate against the card with the provided PIN.
// The PIN is required to use and modify certain slots.
//
// After a specific number of authentication attemps with an invalid PIN,
// usually 3, the PIN will become block and refuse further attempts. At that
// point the PUK must be used to unblock the PIN.
//
// Use DefaultPIN if the PIN hasn't been set.
func (yk *YubiKey) authPIN(pin string) error {
	return ykLogin(yk.scTx, pin)
}

func ykLogin(tx pcsc.Tx, pin string) error {
	data, err := encodePIN(pin)
	if err != nil {
		return err
	}

	// https://csrc.nist.gov/CSRC/media/Publications/sp/800-73/4/archive/2015-05-29/documents/sp800_73-4_pt2_draft.pdf#page=20
	cmd := pcsc.APDU{
		Instruction: insVerify,
		Param2:      0x80,
		Data:        data,
	}
	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("verify pin: %w", err)
	}
	return nil
}

func ykLoginNeeded(tx pcsc.Tx) bool {
	cmd := pcsc.APDU{
		Instruction: insVerify,
		Param2:      0x80,
	}
	_, err := tx.Transmit(cmd)
	return err != nil
}

// Retries returns the number of attempts remaining to enter the correct PIN.
func (yk *YubiKey) Retries() (int, error) {
	return ykPINRetries(yk.scTx)
}

func ykPINRetries(tx pcsc.Tx) (int, error) {
	cmd := pcsc.APDU{
		Instruction: insVerify,
		Param2:      0x80,
	}
	_, err := tx.Transmit(cmd)
	if err == nil {
		return 0, fmt.Errorf("expected error code from empty pin")
	}

	var e *pcsc.AuthErr
	if errors.As(err, &e) {
		return e.Retries, nil
	}

	return 0, fmt.Errorf("invalid response: %w", err)
}

// Reset resets the YubiKey PIV applet to its factory settings, wiping all slots
// and resetting the PIN, PUK, and Management Key to their default values. This
// does NOT affect data on other applets, such as GPG or U2F.
func (yk *YubiKey) Reset() error {
	return ykReset(yk.scTx, yk.rand)
}

func ykReset(tx pcsc.Tx, r io.Reader) error {
	// Reset only works if both the PIN and PUK are blocked. Before resetting,
	// try the wrong PIN and PUK multiple times to block them.

	maxPIN := big.NewInt(100_000_000)
	pinInt, err := rand.Int(r, maxPIN)
	if err != nil {
		return fmt.Errorf("generating random pin: %v", err)
	}
	pukInt, err := rand.Int(r, maxPIN)
	if err != nil {
		return fmt.Errorf("generating random puk: %v", err)
	}

	pin := pinInt.String()
	puk := pukInt.String()

	for {
		err := ykLogin(tx, pin)
		if err == nil {
			// TODO: do we care about a 1/100million chance?
			return fmt.Errorf("expected error with random pin")
		}

		var e *pcsc.AuthErr
		if !errors.As(err, &e) {
			return fmt.Errorf("blocking pin: %w", err)
		}

		if e.Retries == 0 {
			break
		}
	}

	for {
		err := ykChangePUK(tx, puk, puk)
		if err == nil {
			// TODO: do we care about a 1/100million chance?
			return fmt.Errorf("expected error with random puk")
		}

		var e *pcsc.AuthErr
		if !errors.As(err, &e) {
			return fmt.Errorf("blocking puk: %w", err)
		}

		if e.Retries == 0 {
			break
		}
	}

	cmd := pcsc.APDU{
		Instruction: insReset,
	}
	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("reseting yubikey: %w", err)
	}
	return nil
}

type version struct {
	major byte
	minor byte
	patch byte
}

func (v version) String() string {
	return fmt.Sprintf("%d.%d.%d", v.major, v.minor, v.patch)
}

var (
	// Smartcard Application IDs for YubiKeys.
	//
	// https://github.com/Yubico/yubico-piv-tool/blob/yubico-piv-tool-1.7.0/lib/ykpiv.c#L1877
	// https://github.com/Yubico/yubico-piv-tool/blob/yubico-piv-tool-1.7.0/lib/ykpiv.c#L108-L110
	// https://github.com/Yubico/yubico-piv-tool/blob/yubico-piv-tool-1.7.0/lib/ykpiv.c#L1117

	aidManagement = [...]byte{0xa0, 0x00, 0x00, 0x05, 0x27, 0x47, 0x11, 0x17}
	aidPIV        = [...]byte{0xa0, 0x00, 0x00, 0x03, 0x08}
	aidYubiKey    = [...]byte{0xa0, 0x00, 0x00, 0x05, 0x27, 0x20, 0x01, 0x01}
)

func ykAuthenticate(tx pcsc.Tx, key *ManagementKey, rand io.Reader) error {
	// https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-73-4.pdf#page=92
	// https://tsapps.nist.gov/publication/get_pdf.cfm?pub_id=918402#page=114

	// TODO(buglloc): holly shit, need to simplify this crap with generic Tlv struct implementation

	// request a witness
	cmd := pcsc.APDU{
		Instruction: insAuthenticate,
		Param1:      key.Algo.Byte(),
		Param2:      keyCardManagement,
		Data: []byte{
			0x7c, // Dynamic Authentication Template tag
			0x02, // Length of object
			0x80, // 'Witness'
			0x00, // Return encrypted random
		},
	}
	resp, err := tx.Transmit(cmd)
	if err != nil {
		return fmt.Errorf("get auth challenge: %w", err)
	}
	challengeLen := key.Algo.ChallengeLen()
	if n := len(resp); n < 4+challengeLen {
		return fmt.Errorf("challenge didn't return enough bytes: %d", n)
	}

	if !bytes.Equal(resp[:4], []byte{
		0x7c,
		byte(2 + challengeLen),
		0x80,               // 'Witness'
		byte(challengeLen), // Tag length
	}) {
		return fmt.Errorf("invalid authentication object header: %x", resp[:4])
	}

	cardChallenge := resp[4 : 4+challengeLen]
	cardResponse := make([]byte, challengeLen)

	block, err := key.Cipher()
	if err != nil {
		return fmt.Errorf("creating triple des block cipher: %v", err)
	}
	block.Decrypt(cardResponse, cardChallenge)

	challenge := make([]byte, challengeLen)
	if _, err := io.ReadFull(rand, challenge); err != nil {
		return fmt.Errorf("reading rand data: %v", err)
	}
	response := make([]byte, challengeLen)
	block.Encrypt(response, challenge)

	data := []byte{
		0x7c,                     // Dynamic Authentication Template tag
		byte(4 + challengeLen*2), // 2+<challenge>+2+<challenge>
		0x80,                     // 'Witness'
		byte(challengeLen),       // Tag length
	}
	data = append(data, cardResponse...)
	data = append(data,
		0x81,               // 'Challenge'
		byte(challengeLen), // Tag length
	)
	data = append(data, challenge...)

	cmd = pcsc.APDU{
		Instruction: insAuthenticate,
		Param1:      key.Algo.Byte(),
		Param2:      keyCardManagement,
		Data:        data,
	}
	resp, err = tx.Transmit(cmd)
	if err != nil {
		return fmt.Errorf("auth challenge: %w", err)
	}
	if n := len(resp); n < 4+challengeLen {
		return fmt.Errorf("challenge response didn't return enough bytes: %d", n)
	}
	if !bytes.Equal(resp[:4], []byte{
		0x7c,
		byte(2 + challengeLen),
		0x82, // 'Response'
		byte(challengeLen),
	}) {
		return fmt.Errorf("response invalid authentication object header: %x", resp[:4])
	}
	if !bytes.Equal(resp[4:4+challengeLen], response) {
		return fmt.Errorf("challenge failed")
	}

	return nil
}

// ManagementKey returns current management key
func (yk *YubiKey) ManagementKey(pin string) (*ManagementKey, error) {
	protectedMeta, err := yk.ProtectedMetadata(pin)
	if err != nil {
		return nil, fmt.Errorf("unable to get protected metadata: %w", err)
	}

	if len(protectedMeta.ManagementKey) == 0 {
		return nil, ErrNoManagementKey
	}

	if yk.version.major < 5 || yk.version.minor < 3 {
		// management key metadata required 5.3.0+
		return &ManagementKey{
			Key: protectedMeta.ManagementKey,
			ManagementKeyMetadata: ManagementKeyMetadata{
				Algo:        ManagementKeyAlgoTDES,
				TouchPolicy: TouchPolicyNever,
			},
		}, nil
	}

	md, err := ykGetManagementKeyMetadata(yk.scTx)
	if err != nil {
		return nil, err
	}

	return &ManagementKey{
		Key:                   protectedMeta.ManagementKey,
		ManagementKeyMetadata: md,
	}, nil
}

// ykGetManagementKeyMetadata returns management key metadata.
// https://docs.yubico.com/yesdk/users-manual/application-piv/apdu/metadata.html
func ykGetManagementKeyMetadata(tx pcsc.Tx) (ManagementKeyMetadata, error) {
	var out ManagementKeyMetadata

	cmd := pcsc.APDU{
		Instruction: insGetMetadata,
		Param1:      0x00,
		Param2:      keyCardManagement,
	}

	rest, err := tx.Transmit(cmd)
	if err != nil {
		return out, fmt.Errorf("command failed: %w", err)
	}

	for len(rest) > 0 {
		var v asn1.RawValue
		rest, err = asn1.Unmarshal(rest, &v)
		if err != nil {
			return out, fmt.Errorf("unmarshal failed: %w", err)
		}

		switch v.Tag {
		case metadataAlgoTag:
			out.Algo, err = ParseManagementKeyType(v.Bytes)
			if err != nil {
				return out, fmt.Errorf("unable to parse algo: %w", err)
			}
		case metadataPolicyTag:
			if len(v.Bytes) != 2 {
				return out, fmt.Errorf("expected 2 bytes from key policy, got: %d", len(v.Bytes))
			}

			switch v.Bytes[1] {
			case 0x00, 0x01:
				out.TouchPolicy = TouchPolicyNever
			case 0x02, 0x03:
				out.TouchPolicy = TouchPolicyAlways
			default:
				return out, fmt.Errorf("unrecognized touch policy: 0x%x", v.Bytes[1])
			}
		}
	}

	return out, nil
}

// SetManagementKey updates the management key to a new key
func (yk *YubiKey) SetManagementKey(oldKey, newKey *ManagementKey) error {
	if newKey.Algo != ManagementKeyAlgoTDES && yk.version.major < 5 && yk.version.minor < 4 {
		return fmt.Errorf("unable to use %s key algo: %w", newKey.Algo, &VersionErr{
			CurVersion: yk.version.String(),
			MinVersion: "5.4.0",
		})
	}

	if len(newKey.Key) != newKey.Algo.KeyLen() {
		return fmt.Errorf("invalid key: expected %d bytes but got %d", newKey.Algo.KeyLen(), len(newKey.Key))
	}

	if err := ykAuthenticate(yk.scTx, oldKey, yk.rand); err != nil {
		return fmt.Errorf("authenticating with old key: %w", err)
	}

	if err := ykSetManagementKey(yk.scTx, newKey); err != nil {
		return err
	}
	return nil
}

// ykSetManagementKey updates the management key to a new key. This requires
// authenticating with the existing management key.
// https://docs.yubico.com/yesdk/users-manual/application-piv/apdu/set-mgmt-key.html
func ykSetManagementKey(tx pcsc.Tx, key *ManagementKey) error {
	cmd := pcsc.APDU{
		Instruction: insSetMGMKey,
		Param1:      0xff,
		Data: append([]byte{
			key.Algo.Byte(),
			keyCardManagement,
			// it's safe due to we have key up to 32 bytes
			byte(len(key.Key)),
		}, key.Key...),
	}

	switch key.TouchPolicy {
	case TouchPolicyAlways, TouchPolicyCached:
		cmd.Param2 = 0xfe
	default:
		cmd.Param2 = 0xff
	}

	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("command failed: %w", err)
	}
	return nil
}

// SetPIN updates the PIN to a new value. For compatibility, PINs should be 1-8
// numeric characters.
//
// To generate a new PIN, use the crypto/rand package.
//
//		// Generate a 6 character PIN.
//		newPINInt, err := rand.Int(rand.Reader, bit.NewInt(1_000_000))
//		if err != nil {
//			// ...
//		}
//		// Format with leading zeros.
//		newPIN := fmt.Sprintf("%06d", newPINInt)
//		if err := yk.SetPIN(piv.DefaultPIN, newPIN); err != nil {
//			// ...
//		}
//
func (yk *YubiKey) SetPIN(oldPIN, newPIN string) error {
	return ykChangePIN(yk.scTx, oldPIN, newPIN)
}

func ykChangePIN(tx pcsc.Tx, oldPIN, newPIN string) error {
	oldPINData, err := encodePIN(oldPIN)
	if err != nil {
		return fmt.Errorf("encoding old pin: %v", err)
	}
	newPINData, err := encodePIN(newPIN)
	if err != nil {
		return fmt.Errorf("encoding new pin: %v", err)
	}
	cmd := pcsc.APDU{
		Instruction: insChangeReference,
		Param2:      0x80,
		Data:        append(oldPINData, newPINData...),
	}
	_, err = tx.Transmit(cmd)
	return err
}

// Unblock unblocks the PIN, setting it to a new value.
func (yk *YubiKey) Unblock(puk, newPIN string) error {
	return ykUnblockPIN(yk.scTx, puk, newPIN)
}

func ykUnblockPIN(tx pcsc.Tx, puk, newPIN string) error {
	pukData, err := encodePIN(puk)
	if err != nil {
		return fmt.Errorf("encoding puk: %v", err)
	}
	newPINData, err := encodePIN(newPIN)
	if err != nil {
		return fmt.Errorf("encoding new pin: %v", err)
	}
	cmd := pcsc.APDU{
		Instruction: insResetRetry,
		Param2:      0x80,
		Data:        append(pukData, newPINData...),
	}
	_, err = tx.Transmit(cmd)
	return err
}

// SetPUK updates the PUK to a new value. For compatibility, PUKs should be 1-8
// numeric characters.
//
// To generate a new PUK, use the crypto/rand package.
//
//		// Generate a 8 character PUK.
//		newPUKInt, err := rand.Int(rand.Reader, bit.NewInt(100_000_000))
//		if err != nil {
//			// ...
//		}
//		// Format with leading zeros.
//		newPUK := fmt.Sprintf("%08d", newPUKInt)
//		if err := yk.SetPIN(piv.DefaultPUK, newPUK); err != nil {
//			// ...
//		}
//
func (yk *YubiKey) SetPUK(oldPUK, newPUK string) error {
	return ykChangePUK(yk.scTx, oldPUK, newPUK)
}

func ykChangePUK(tx pcsc.Tx, oldPUK, newPUK string) error {
	oldPUKData, err := encodePIN(oldPUK)
	if err != nil {
		return fmt.Errorf("encoding old puk: %v", err)
	}
	newPUKData, err := encodePIN(newPUK)
	if err != nil {
		return fmt.Errorf("encoding new puk: %v", err)
	}
	cmd := pcsc.APDU{
		Instruction: insChangeReference,
		Param2:      0x81,
		Data:        append(oldPUKData, newPUKData...),
	}
	_, err = tx.Transmit(cmd)
	return err
}

func ykSelectApplication(tx pcsc.Tx, id []byte) error {
	cmd := pcsc.APDU{
		Instruction: insSelectApplication,
		Param1:      0x04,
		Data:        id[:],
	}
	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("command failed: %w", err)
	}
	return nil
}

func ykVersion(tx pcsc.Tx) (*version, error) {
	cmd := pcsc.APDU{
		Instruction: insGetVersion,
	}
	resp, err := tx.Transmit(cmd)
	if err != nil {
		return nil, fmt.Errorf("command failed: %w", err)
	}
	if n := len(resp); n != 3 {
		return nil, fmt.Errorf("expected response to have 3 bytes, got: %d", n)
	}
	return &version{resp[0], resp[1], resp[2]}, nil
}

func ykSerial(tx pcsc.Tx, v *version) (uint32, error) {
	cmd := pcsc.APDU{
		Instruction: insGetSerial,
	}

	if v.major < 5 {
		// Earlier versions of YubiKeys required using the yubikey applet to get
		// the serial number. Newer ones have this built into the PIV applet.
		if err := ykSelectApplication(tx, aidYubiKey[:]); err != nil {
			return 0, fmt.Errorf("selecting yubikey applet: %w", err)
		}
		defer func() { _ = ykSelectApplication(tx, aidPIV[:]) }()

		cmd = pcsc.APDU{
			Instruction: 0x01,
			Param1:      0x10,
		}
	}

	resp, err := tx.Transmit(cmd)
	if err != nil {
		return 0, fmt.Errorf("smart card command: %w", err)
	}
	if n := len(resp); n != 4 {
		return 0, fmt.Errorf("expected 4 byte serial number, got %d", n)
	}
	return binary.BigEndian.Uint32(resp), nil
}

// ProtectedMetadata returns protected data stored on the card. This can be used to
// retrieve PIN protected management keys.
func (yk *YubiKey) ProtectedMetadata(pin string) (*ProtectedMetadata, error) {
	m, err := ykGetProtectedMetadata(yk.scTx, pin)
	if err != nil {
		if errors.Is(err, pcsc.ErrNotFound) {
			return &ProtectedMetadata{}, nil
		}
		return nil, err
	}
	return m, nil
}

// SetProtectedMetadata sets PIN protected metadata on the key. This is primarily to
// store the management key on the smart card instead of managing the PIN and
// management key seperately.
func (yk *YubiKey) SetProtectedMetadata(key *ManagementKey, m *ProtectedMetadata) error {
	return ykSetProtectedMetadata(yk.scTx, key, m)
}

// Metadata returns unprotected metadata stored on the card.
func (yk *YubiKey) Metadata() (*Metadata, error) {
	m, err := ykGetMetadata(yk.scTx)
	if err != nil {
		if errors.Is(err, pcsc.ErrNotFound) {
			return &Metadata{}, nil
		}
		return nil, err
	}
	return m, nil
}

// SetMetadata sets unprotected metadata on the key.
func (yk *YubiKey) SetMetadata(m *Metadata) error {
	return ykSetMetadata(yk.scTx, m)
}

// ProtectedMetadata holds protected metadata. This is primarily used by YubiKey manager
// to implement PIN protect management keys, storing management keys on the card
// guarded by the PIN.
type ProtectedMetadata struct {
	// ManagementKey is the management key stored directly on the YubiKey.
	ManagementKey []byte

	// raw, if not nil, is the full bytes
	raw []byte
}

func (m *ProtectedMetadata) Marshal() ([]byte, error) {
	if m.raw == nil {
		if len(m.ManagementKey) == 0 {
			return []byte{0x88, 0x00}, nil
		}

		keyLen := byte(len(m.ManagementKey))
		return append([]byte{
			0x88,
			2 + keyLen,
			0x89,
			keyLen,
		}, m.ManagementKey...), nil
	}

	if len(m.ManagementKey) == 0 {
		return m.raw, nil
	}

	var metadata asn1.RawValue
	if _, err := asn1.Unmarshal(m.raw, &metadata); err != nil {
		return nil, fmt.Errorf("updating metadata: %v", err)
	}
	if !bytes.HasPrefix(metadata.FullBytes, []byte{0x88}) {
		return nil, fmt.Errorf("expected tag: 0x88")
	}
	raw := metadata.Bytes

	metadata.Bytes = nil
	metadata.FullBytes = nil

	for len(raw) > 0 {
		var (
			err error
			v   asn1.RawValue
		)
		raw, err = asn1.Unmarshal(raw, &v)
		if err != nil {
			return nil, fmt.Errorf("unmarshal metadata field: %v", err)
		}

		if bytes.HasPrefix(v.FullBytes, []byte{0x89}) {
			continue
		}
		metadata.Bytes = append(metadata.Bytes, v.FullBytes...)
	}
	metadata.Bytes = append(metadata.Bytes, 0x89, byte(len(m.ManagementKey)))
	metadata.Bytes = append(metadata.Bytes, m.ManagementKey...)
	return asn1.Marshal(metadata)
}

func (m *ProtectedMetadata) Unmarshal(b []byte) error {
	m.raw = b
	var md asn1.RawValue
	if _, err := asn1.Unmarshal(b, &md); err != nil {
		return err
	}
	if !bytes.HasPrefix(md.FullBytes, []byte{0x88}) {
		return fmt.Errorf("expected tag: 0x88")
	}
	d := md.Bytes
	for len(d) > 0 {
		var (
			err error
			v   asn1.RawValue
		)
		d, err = asn1.Unmarshal(d, &v)
		if err != nil {
			return fmt.Errorf("unmarshal metadata field: %v", err)
		}

		// 0x89 indicates key
		if !bytes.HasPrefix(v.FullBytes, []byte{0x89}) {
			continue
		}

		m.ManagementKey = make([]byte, len(v.Bytes))
		copy(m.ManagementKey, v.Bytes)
	}
	return nil
}

func ykGetProtectedMetadata(tx pcsc.Tx, pin string) (*ProtectedMetadata, error) {
	// NOTE: for some reason this action requires the PIN to be authenticated on
	// the same transaction. It doesn't work otherwise.
	if err := ykLogin(tx, pin); err != nil {
		return nil, fmt.Errorf("authenticating with pin: %w", err)
	}
	cmd := pcsc.APDU{
		Instruction: insGetData,
		Param1:      0x3f,
		Param2:      0xff,
		Data: []byte{
			0x5c, // Tag list
			0x03,
			0x5f,
			0xc1,
			0x09,
		},
	}
	resp, err := tx.Transmit(cmd)
	if err != nil {
		return nil, fmt.Errorf("command failed: %w", err)
	}
	obj, _, err := unmarshalASN1(resp, 1, 0x13) // tag 0x53
	if err != nil {
		return nil, fmt.Errorf("unmarshaling response: %v", err)
	}
	var m ProtectedMetadata
	if err := m.Unmarshal(obj); err != nil {
		return nil, fmt.Errorf("unmarshal protected metadata: %v", err)
	}
	return &m, nil
}

func ykSetProtectedMetadata(tx pcsc.Tx, key *ManagementKey, m *ProtectedMetadata) error {
	data, err := m.Marshal()
	if err != nil {
		return fmt.Errorf("encoding metadata: %v", err)
	}
	data = append([]byte{
		0x5c, // Tag list
		0x03,
		0x5f,
		0xc1,
		0x09,
	}, marshalASN1(0x53, data)...)
	cmd := pcsc.APDU{
		Instruction: insPutData,
		Param1:      0x3f,
		Param2:      0xff,
		Data:        data,
	}
	// NOTE: for some reason this action requires the management key authenticated
	// on the same transaction. It doesn't work otherwise.
	if err := ykAuthenticate(tx, key, rand.Reader); err != nil {
		return fmt.Errorf("authenticating with key: %w", err)
	}
	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("command failed: %w", err)
	}
	return nil
}

// Metadata holds metadata. This is primarily used by YubiKey manager to store various flags/salt about yubikey.
// Reference: https://github.com/Yubico/yubikey-manager/blob/e46bd4f8b327c986a81b142b2f0455db755eb606/ykman/piv.py#L164-L216
type Metadata struct {
	Flags        uint8
	Salt         []byte
	PinTimestamp uint32

	// raw, if not nil, is the full bytes
	raw []byte
}

const (
	MetadataFlagPukBlocked      = 0x01
	MetadataFlagMgmKeyProtected = 0x02

	metadataFlagsTag = 0x81
	metadataSaltTag  = 0x82
	metadataPinTSTag = 0x83
)

func (m *Metadata) Marshal() ([]byte, error) {
	metadata := asn1.RawValue{
		Tag:        0x00,
		IsCompound: false,
		Class:      0x02,
	}
	if m.raw != nil {
		if _, err := asn1.Unmarshal(m.raw, &metadata); err != nil {
			return nil, fmt.Errorf("updating metadata: %v", err)
		}
		if !bytes.HasPrefix(metadata.FullBytes, []byte{0x80}) {
			return nil, fmt.Errorf("expected tag: 0x80")
		}

		raw := metadata.Bytes
		metadata.Bytes = nil
		metadata.FullBytes = nil

		for len(raw) > 0 {
			var (
				err error
				v   asn1.RawValue
			)
			raw, err = asn1.Unmarshal(raw, &v)
			if err != nil {
				return nil, fmt.Errorf("unmarshal metadata field: %v", err)
			}

			skip := false
			switch v.FullBytes[0] {
			case metadataFlagsTag, metadataSaltTag, metadataPinTSTag:
				skip = true
			}
			if skip {
				continue
			}

			metadata.Bytes = append(metadata.Bytes, v.FullBytes...)
		}
	}

	metadata.Bytes = append(metadata.Bytes, metadataFlagsTag, 1, m.Flags)

	if len(m.Salt) > 0 {
		metadata.Bytes = append(metadata.Bytes, metadataSaltTag, uint8(len(m.Salt)))
		metadata.Bytes = append(metadata.Bytes, m.Salt...)
	}

	if m.PinTimestamp > 0 {
		data := make([]byte, 4)
		binary.BigEndian.PutUint32(data, m.PinTimestamp)
		metadata.Bytes = append(metadata.Bytes, metadataPinTSTag, 4)
		metadata.Bytes = append(metadata.Bytes, data...)
	}

	return asn1.Marshal(metadata)
}

func (m *Metadata) Unmarshal(b []byte) error {
	m.raw = b
	var md asn1.RawValue
	if _, err := asn1.Unmarshal(b, &md); err != nil {
		return err
	}
	if !bytes.HasPrefix(md.FullBytes, []byte{0x80}) {
		return fmt.Errorf("expected tag: 0x80")
	}
	d := md.Bytes
	for len(d) > 0 {
		var (
			err error
			v   asn1.RawValue
		)
		d, err = asn1.Unmarshal(d, &v)
		if err != nil {
			return fmt.Errorf("unmarshal metadata field: %v", err)
		}

		switch v.FullBytes[0] {
		case metadataFlagsTag:
			if len(v.Bytes) != 1 {
				return fmt.Errorf("invalid flags size: %d", len(v.Bytes))
			}
			m.Flags = v.Bytes[0]
		case metadataSaltTag:
			m.Salt = v.Bytes
		case metadataPinTSTag:
			if len(v.Bytes) != 4 {
				return fmt.Errorf("invalid pin_timestamp size: %d", len(v.Bytes))
			}
			m.PinTimestamp = binary.BigEndian.Uint32(v.Bytes)
		}
	}
	return nil
}

func ykGetMetadata(tx pcsc.Tx) (*Metadata, error) {
	cmd := pcsc.APDU{
		Instruction: insGetData,
		Param1:      0x3f,
		Param2:      0xff,
		Data: []byte{
			0x5c, // Tag list
			0x03,
			0x5f,
			0xff,
			0x00,
		},
	}

	resp, err := tx.Transmit(cmd)
	if err != nil {
		return nil, fmt.Errorf("command failed: %w", err)
	}
	obj, _, err := unmarshalASN1(resp, 1, 0x13) // tag 0x53
	if err != nil {
		return nil, fmt.Errorf("unmarshaling response: %v", err)
	}
	var m Metadata
	if err := m.Unmarshal(obj); err != nil {
		return nil, fmt.Errorf("unmarshal metadata: %v", err)
	}
	return &m, nil
}

func ykSetMetadata(tx pcsc.Tx, m *Metadata) error {
	data, err := m.Marshal()
	if err != nil {
		return fmt.Errorf("encoding metadata: %v", err)
	}
	data = append([]byte{
		0x5c, // Tag list
		0x03,
		0x5f,
		0xff,
		0x00,
	}, marshalASN1(0x53, data)...)
	cmd := pcsc.APDU{
		Instruction: insPutData,
		Param1:      0x3f,
		Param2:      0xff,
		Data:        data,
	}
	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("command failed: %w", err)
	}
	return nil
}
