package porto

import (
	"fmt"
)

type (
	API struct {
		porto *Connection
	}

	APIOpts struct {
		// Maximum connections in pool (default=10)
		MaxConnections int
	}

	CreateVolumeOpts struct {
		//
		Path string
	}
)

var (
	DefaultAPIOpts = APIOpts{
		MaxConnections: 10,
	}
)

func (a *APIOpts) PortoConnectionOpts() ConnectionOpts {
	return ConnectionOpts{
		MaxConnections: a.MaxConnections,
	}
}

// Create new API
func NewAPI(opts *APIOpts) (api *API, err error) {
	var porto *Connection
	if opts == nil {
		porto, err = NewPortoConnection(DefaultAPIOpts.PortoConnectionOpts())
	} else {
		porto, err = NewPortoConnection(opts.PortoConnectionOpts())
	}

	if err != nil {
		return nil, err
	}

	return &API{
		porto: porto,
	}, nil
}

// Returns proto connection
func (a *API) Connection() *Connection {
	return a.porto
}

// Close API connections
func (a *API) Close() error {
	err := a.porto.Close()
	if err != nil {
		return fmt.Errorf("failed to close: %s", err.Error())
	}
	return nil
}

// Generic API

// Get porto version
func (a *API) Version() (version string, err error) {
	version, _, err = a.porto.GetVersion()
	if err != nil {
		err = fmt.Errorf("failed to get version: %s", err.Error())
		return
	}
	return
}

// Get porto revision
func (a *API) Revision() (revision string, err error) {
	_, revision, err = a.porto.GetVersion()
	if err != nil {
		err = fmt.Errorf("failed to get revision: %s", err.Error())
		return
	}
	return
}

// Containers API

// Create stopped container
func (a *API) CreateContainer(name string) (container Container, err error) {
	err = a.porto.Create(name)
	if err != nil {
		err = fmt.Errorf("failed to create container: %s", err.Error())
		return
	}

	container = Container{
		name:  name,
		porto: a.porto,
	}
	return
}

// Create weak stopped container
func (a *API) CreateWeakContainer(name string) (container Container, err error) {
	err = a.porto.CreateWeak(name)
	if err != nil {
		err = fmt.Errorf("failed to create container: %s", err.Error())
		return
	}

	container = Container{
		name:  name,
		porto: a.porto,
	}
	return
}

// Get container by name
func (a *API) GetContainer(name string) (container Container, err error) {
	_, err = a.porto.GetProperty(name, "state")
	if err != nil {
		err = fmt.Errorf("failed to get container '%s': %s", name, err.Error())
		return
	}

	container = Container{
		name:  name,
		porto: a.porto,
	}
	return
}

// Get container by pid
func (a *API) GetContainerByPid(pid uint32) (container Container, err error) {
	name, err := a.porto.LocateProcess(pid, "")
	if err != nil {
		err = fmt.Errorf("failed to get container by pid '%d': %s", pid, err.Error())
		return
	}

	container = Container{
		name:  name,
		porto: a.porto,
	}
	return
}

// Get container by pid with comm
func (a *API) GetContainerByPidWithComm(pid uint32, comm string) (container Container, err error) {
	name, err := a.porto.LocateProcess(pid, comm)
	if err != nil {
		err = fmt.Errorf("failed to get container by pid '%d' with comm '%s': %s", pid, comm, err.Error())
		return
	}

	container = Container{
		name:  name,
		porto: a.porto,
	}
	return
}

// List containers
func (a *API) ListContainers() ([]Container, error) {
	return a.ListContainersByMask("")
}

// List containers by name mask
func (a *API) ListContainersByMask(mask string) ([]Container, error) {
	containers, err := a.porto.ListByMask(mask)
	if err != nil {
		return nil, fmt.Errorf("failed to list containers: %s", err.Error())
	}
	result := make([]Container, len(containers))
	for i, name := range containers {
		result[i] = Container{
			name:  name,
			porto: a.porto,
		}
	}

	return result, nil
}

// List available properties
func (a *API) ListAvailableProperties() ([]AvailableProperty, error) {
	properties, err := a.porto.ListProperties()
	if err != nil {
		return properties, fmt.Errorf("failed to list properties: %s", err.Error())
	}
	return properties, nil
}

// TODO(buglloc): Volumes API
// Creates new volume
func (a *API) CreateVolume(path string, config map[string]string) (volume *Volume, resultErr error) {
	desc, err := a.porto.CreateVolume(path, config)
	if err != nil {
		resultErr = err
		return
	}

	volume = &Volume{
		path:  desc.Path,
		porto: a.porto,
	}
	return
}

// Returns existed volume
func (a *API) GetVolume(path string) (volume *Volume, resultErr error) {
	desc, err := a.porto.GetVolume(path, "")
	if err != nil {
		resultErr = err
		return
	}

	volume = &Volume{
		path:  desc.Path,
		porto: a.porto,
	}
	return
}

// List available volumes
func (a *API) ListVolumes(opts ListVolumesOpts) ([]*Volume, error) {
	descs, err := a.porto.ListVolumes(opts.Path, opts.Container)
	if err != nil {
		return nil, err
	}

	volumes := make([]*Volume, len(descs))
	for i, d := range descs {
		volumes[i] = &Volume{
			path:  d.Path,
			porto: a.porto,
		}
	}
	return volumes, nil
}

func (a *API) ImportLayer(opts ImportLayerOpts) error {
	return a.porto.ImportLayer(opts)
}

func (a *API) RemoveLayer(layerID string) error {
	return a.porto.RemoveLayer(layerID)
}

func (a *API) ListLayers(opts ListLayersOpts) ([]LayerDescription, error) {
	return a.porto.ListLayers2(opts.Place, opts.Mask)
}
