package porto

import (
	"fmt"
	"os"
	"strings"
	"syscall"
	"time"

	"a.yandex-team.ru/security/libs/go/porto/internal/strutils"
)

type (
	Container struct {
		name  string
		porto *Connection
	}
)

// Get container name
func (c *Container) GetName() string {
	return c.name
}

// Start stopped container
func (c *Container) Start() error {
	err := c.porto.Start(c.name)
	if err != nil {
		return fmt.Errorf("failed to start container '%s': %s", c.name, err.Error())
	}

	return nil
}

// Stop dead or running container
func (c *Container) Stop() error {
	err := c.porto.Stop(c.name)
	if err != nil {
		return fmt.Errorf("failed to stop container '%s': %s", c.name, err.Error())
	}

	return nil
}

// Stop dead or running container with wait timeout
func (c *Container) StopWaitD(timeout time.Duration) error {
	err := c.porto.StopWithTimeout(c.name, timeout)
	if err != nil {
		return fmt.Errorf("failed to stop container '%s': %s", c.name, err.Error())
	}

	return nil
}

// Freeze running container
func (c *Container) Pause() error {
	err := c.porto.Pause(c.name)
	if err != nil {
		return fmt.Errorf("failed to pause container '%s': %s", c.name, err.Error())
	}

	return nil
}

// Unfreeze paused container
func (c *Container) Resume() error {
	err := c.porto.Resume(c.name)
	if err != nil {
		return fmt.Errorf("failed to resume container '%s': %s", c.name, err.Error())
	}

	return nil
}

// Stop and destroy container
func (c *Container) Destroy() error {
	err := c.porto.Destroy(c.name)
	if err != nil {
		return fmt.Errorf("failed to destroy container '%s': %s", c.name, err.Error())
	}

	return nil
}

// Send signal main process in container
func (c *Container) Kill(sig syscall.Signal) error {
	err := c.porto.Kill(c.name, sig)
	if err != nil {
		return fmt.Errorf("failed to kill container '%s' with '%v': %s", c.name, sig, err.Error())
	}

	return nil
}

// Wait for container death
func (c *Container) Wait(timeout time.Duration) (bool, error) {
	name, err := c.porto.Wait(c.name, timeout)
	if err != nil {
		return false, fmt.Errorf("failed to wait container '%s': %s", c.name, err.Error())
	}

	return name != "", nil
}

// Get container property
func (c *Container) GetProperty(name string) (string, error) {
	value, err := c.porto.GetProperty(c.name, name)
	if err != nil {
		return "", fmt.Errorf("failed to get container '%s' property '%s': %s", c.name, name, err.Error())
	}

	return value, nil
}

// Set container property
func (c *Container) SetProperty(name, value string) error {
	err := c.porto.SetProperty(c.name, name, value)
	if err != nil {
		return fmt.Errorf("failed to set container '%s' property '%s': %s", c.name, name, err.Error())
	}

	return nil
}

// Get container label
func (c *Container) GetLabel() (string, error) {
	value, err := c.porto.GetProperty(c.name, "labels")
	if err != nil {
		return "", fmt.Errorf("failed to get container '%s' label: %s", c.name, err.Error())
	}

	return value, nil
}

// Set container label
func (c *Container) SetLabel(label string) error {
	err := c.porto.SetProperty(c.name, "labels", label)
	if err != nil {
		return fmt.Errorf("failed to set container '%s' label: %s", c.name, err.Error())
	}

	return nil
}

// Get environment variables of main container process (from porto side, not real!)
func (c *Container) GetEnv() (map[string]string, error) {
	value, err := c.porto.GetProperty(c.name, "env")
	if err != nil {
		return nil, fmt.Errorf("failed to get container '%s' environment: %s", c.name, err.Error())
	}

	envs := strutils.SplitQuoted(value, ';')
	result := make(map[string]string, len(envs))
	for _, env := range envs {
		values := strings.SplitN(env, "=", 2)
		if len(values) != 2 {
			return nil, fmt.Errorf("failed to parse container '%s' environment variable: %s", c.name, env)
		}

		result[values[0]] = values[1]
	}

	return result, nil
}

// Set environment variables of main container process
// Container with isolate=false inherits environment variables from parent.
func (c *Container) SetEnv(env map[string]string) error {
	envEscaper := strings.NewReplacer(
		`\`, `\\`,
		";", `\;`,
	)

	var envBuilder strings.Builder
	for key, value := range env {
		_, _ = envEscaper.WriteString(&envBuilder, key)
		envBuilder.WriteRune('=')
		_, _ = envEscaper.WriteString(&envBuilder, value)
		envBuilder.WriteRune(';')
	}
	err := c.porto.SetProperty(c.name, "env", envBuilder.String())
	if err != nil {
		return fmt.Errorf("failed to set container '%s' environment: %s", c.name, err.Error())
	}

	return nil
}

// Get capabilities boundary set
func (c *Container) GetCapabilities() (CapabilityMask, error) {
	value, err := c.porto.GetProperty(c.name, "capabilities")
	if err != nil {
		return 0, fmt.Errorf("failed to get container '%s' capabilities: %s", c.name, err.Error())
	}

	return CapabilityMaskParse(value)
}

// Get resulting set of capabilities allowed in container
func (c *Container) GetAllowedCapabilities() (CapabilityMask, error) {
	value, err := c.porto.GetProperty(c.name, "capabilities_allowed")
	if err != nil {
		return 0, fmt.Errorf("failed to get container '%s' allowed capabilities: %s", c.name, err.Error())
	}

	return CapabilityMaskParse(value)
}

// Set capabilities boundary set
func (c *Container) SetCapabilities(caps CapabilityMask) error {
	err := c.porto.SetProperty(c.name, "capabilities", caps.String())
	if err != nil {
		return fmt.Errorf("failed to set container '%s' capabilities: %s", c.name, err.Error())
	}

	return nil
}

// Get ambient capabilities
func (c *Container) GetAmbientCapabilities() (CapabilityMask, error) {
	value, err := c.porto.GetProperty(c.name, "capabilities_ambient")
	if err != nil {
		return 0, fmt.Errorf("failed to get container '%s' ambient capabilities: %s", c.name, err.Error())
	}

	return CapabilityMaskParse(value)
}

// Get resulting set of ambient capabilities allowed in container
func (c *Container) GetAllowedAmbientCapabilities() (CapabilityMask, error) {
	value, err := c.porto.GetProperty(c.name, "capabilities_ambient_allowed")
	if err != nil {
		return 0, fmt.Errorf("failed to get container '%s' allowed ambient capabilities: %s", c.name, err.Error())
	}

	return CapabilityMaskParse(value)
}

// Raise ambient capabilities
func (c *Container) SetAmbientCapabilities(caps CapabilityMask) error {
	err := c.porto.SetProperty(c.name, "capabilities_ambient", caps.String())
	if err != nil {
		return fmt.Errorf("failed to set container '%s' ambient capabilities: %s", c.name, err.Error())
	}

	return nil
}

// Get virtualization mode
func (c *Container) GetVirtMode() (VirtMode, error) {
	value, err := c.porto.GetProperty(c.name, "virt_mode")
	if err != nil {
		return 0, fmt.Errorf("failed to get container '%s' virt_mode: %s", c.name, err.Error())
	}

	return VirtModeParse(value)
}

// Set virtualization mode
func (c *Container) SetVirtMode(mode VirtMode) error {
	err := c.porto.SetProperty(c.name, "virt_mode", mode.String())
	if err != nil {
		return fmt.Errorf("failed to set container '%s' virt_mode: %s", c.name, err.Error())
	}

	return nil
}

// Get porto access level
func (c *Container) GetPortoAccess() (PortoAccess, error) {
	value, err := c.porto.GetProperty(c.name, "enable_porto")
	if err != nil {
		return 0, fmt.Errorf("failed to get container '%s' enable_porto: %s", c.name, err.Error())
	}

	return PortoAccessParse(value)
}

// Set porto access level
func (c *Container) SetPortoAccess(mode PortoAccess) error {
	err := c.porto.SetProperty(c.name, "enable_porto", mode.String())
	if err != nil {
		return fmt.Errorf("failed to set container '%s' enable_porto: %s", c.name, err.Error())
	}

	return nil
}

// Check if container is weak
func (c *Container) IsWeak() (bool, error) {
	value, err := c.porto.GetProperty(c.name, "weak")
	if err != nil {
		return false, fmt.Errorf("failed to check if container '%s' is weak: %s", c.name, err.Error())
	}

	return value == "true", nil
}

// Get container state
func (c *Container) GetState() (State, error) {
	value, err := c.porto.GetProperty(c.name, "state")
	if err != nil {
		return StateUnknown, fmt.Errorf("failed to get container '%s' state: %s", c.name, err.Error())
	}

	return StateParse(value)
}

// Move process into container
func (c *Container) AttachProcess(pid uint32) error {
	return c.AttachProcessWithComm(pid, "")
}

// Move process into container
func (c *Container) AttachProcessWithComm(pid uint32, comm string) error {
	err := c.porto.AttachProcess(c.name, pid, comm)
	if err != nil {
		return fmt.Errorf("failed to move process %d into container '%s': %s", pid, c.name, err.Error())
	}

	return nil
}

// Move thread into container
func (c *Container) AttachThread(pid uint32) error {
	return c.AttachThreadWithComm(pid, "")
}

// Move thread into container
func (c *Container) AttachThreadWithComm(pid uint32, comm string) error {
	err := c.porto.AttachThread(c.name, pid, comm)
	if err != nil {
		return fmt.Errorf("failed to move thread %d into container '%s': %s", pid, c.name, err.Error())
	}

	return nil
}

// Creates new stdout pipe
func (c *Container) NewStdoutPipe() (r *os.File, w *os.File, err error) {
	r, w, err = os.Pipe()
	if err != nil {
		return
	}

	err = c.SetProperty("stdout_path", fmt.Sprintf("/dev/fd/%d", w.Fd()))
	if err != nil {
		_ = r.Close()
		_ = w.Close()
		return
	}
	return
}

// Creates new stderr pipe
func (c *Container) NewStderrPipe() (r *os.File, w *os.File, err error) {
	r, w, err = os.Pipe()
	if err != nil {
		return
	}

	err = c.SetProperty("stderr_path", fmt.Sprintf("/dev/fd/%d", w.Fd()))
	if err != nil {
		_ = r.Close()
		_ = w.Close()
		return
	}
	return
}

// Returns container volume
func (c *Container) GetVolume(path string) (volume *Volume, resultErr error) {
	desc, err := c.porto.GetVolume(path, c.name)
	if err != nil {
		resultErr = err
		return
	}

	volume = &Volume{
		path:      desc.Path,
		container: c.GetName(),
		porto:     c.porto,
	}
	return
}

// List available volumes
func (c *Container) ListVolumes() (volumes []*Volume, resultErr error) {
	descs, err := c.porto.ListVolumes("", c.name)
	if err != nil {
		resultErr = err
		return
	}

	for _, d := range descs {
		volumes = append(volumes, &Volume{
			path:      d.Path,
			container: c.GetName(),
			porto:     c.porto,
		})
	}
	return
}
