package porto

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/security/libs/go/porto/internal/strutils"
)

type (
	CapabilityMask uint64
)

const (
	CapChown CapabilityMask = 1 << iota
	CapDacOverride
	CapDacReadSearch
	CapFowner
	CapFsetid
	CapKill
	CapSetgid
	CapSetuid
	CapSetpcap
	CapLinuxImmutable
	CapNetBindService
	CapNetBroadcast
	CapNetAdmin
	CapNetRaw
	CapIpcLock
	CapIpcOwner
	CapSysModule
	CapSysRawio
	CapSysChroot
	CapSysPtrace
	CapSysPacct
	CapSysAdmin
	CapSysBoot
	CapSysNice
	CapSysResource
	CapSysTime
	CapSysTtyConfig
	CapMknod
	CapLease
	CapAuditRead
	CapAuditWrite
	CapAuditControl
	CapSetfcap
	CapMacOverride
	CapMacAdmin
	CapSyslog
	CapWakeAlarm
	CapBlockSuspend
)

func (c CapabilityMask) String() string {
	var result strings.Builder
	if (c & CapChown) != 0 {
		result.WriteString("CHOWN")
		result.WriteByte(';')
	}

	if (c & CapDacOverride) != 0 {
		result.WriteString("DAC_OVERRIDE")
		result.WriteByte(';')
	}

	if (c & CapDacReadSearch) != 0 {
		result.WriteString("DAC_READ_SEARCH")
		result.WriteByte(';')
	}

	if (c & CapFowner) != 0 {
		result.WriteString("FOWNER")
		result.WriteByte(';')
	}

	if (c & CapFsetid) != 0 {
		result.WriteString("FSETID")
		result.WriteByte(';')
	}

	if (c & CapKill) != 0 {
		result.WriteString("KILL")
		result.WriteByte(';')
	}

	if (c & CapSetgid) != 0 {
		result.WriteString("SETGID")
		result.WriteByte(';')
	}

	if (c & CapSetuid) != 0 {
		result.WriteString("SETUID")
		result.WriteByte(';')
	}

	if (c & CapSetpcap) != 0 {
		result.WriteString("SETPCAP")
		result.WriteByte(';')
	}

	if (c & CapLinuxImmutable) != 0 {
		result.WriteString("LINUX_IMMUTABLE")
		result.WriteByte(';')
	}

	if (c & CapNetBindService) != 0 {
		result.WriteString("NET_BIND_SERVICE")
		result.WriteByte(';')
	}

	if (c & CapNetBroadcast) != 0 {
		result.WriteString("NET_BROADCAST")
		result.WriteByte(';')
	}

	if (c & CapNetAdmin) != 0 {
		result.WriteString("NET_ADMIN")
		result.WriteByte(';')
	}

	if (c & CapNetRaw) != 0 {
		result.WriteString("NET_RAW")
		result.WriteByte(';')
	}

	if (c & CapIpcLock) != 0 {
		result.WriteString("IPC_LOCK")
		result.WriteByte(';')
	}

	if (c & CapIpcOwner) != 0 {
		result.WriteString("IPC_OWNER")
		result.WriteByte(';')
	}

	if (c & CapSysModule) != 0 {
		result.WriteString("SYS_MODULE")
		result.WriteByte(';')
	}

	if (c & CapSysRawio) != 0 {
		result.WriteString("SYS_RAWIO")
		result.WriteByte(';')
	}

	if (c & CapSysChroot) != 0 {
		result.WriteString("SYS_CHROOT")
		result.WriteByte(';')
	}

	if (c & CapSysPtrace) != 0 {
		result.WriteString("SYS_PTRACE")
		result.WriteByte(';')
	}

	if (c & CapSysPacct) != 0 {
		result.WriteString("SYS_PACCT")
		result.WriteByte(';')
	}

	if (c & CapSysAdmin) != 0 {
		result.WriteString("SYS_ADMIN")
		result.WriteByte(';')
	}

	if (c & CapSysBoot) != 0 {
		result.WriteString("SYS_BOOT")
		result.WriteByte(';')
	}

	if (c & CapSysNice) != 0 {
		result.WriteString("SYS_NICE")
		result.WriteByte(';')
	}

	if (c & CapSysResource) != 0 {
		result.WriteString("SYS_RESOURCE")
		result.WriteByte(';')
	}

	if (c & CapSysTime) != 0 {
		result.WriteString("SYS_TIME")
		result.WriteByte(';')
	}

	if (c & CapSysTtyConfig) != 0 {
		result.WriteString("SYS_TTY_CONFIG")
		result.WriteByte(';')
	}

	if (c & CapMknod) != 0 {
		result.WriteString("MKNOD")
		result.WriteByte(';')
	}

	if (c & CapLease) != 0 {
		result.WriteString("LEASE")
		result.WriteByte(';')
	}

	if (c & CapAuditRead) != 0 {
		result.WriteString("AUDIT_READ")
		result.WriteByte(';')
	}

	if (c & CapAuditWrite) != 0 {
		result.WriteString("AUDIT_WRITE")
		result.WriteByte(';')
	}

	if (c & CapAuditControl) != 0 {
		result.WriteString("AUDIT_CONTROL")
		result.WriteByte(';')
	}

	if (c & CapSetfcap) != 0 {
		result.WriteString("SETFCAP")
		result.WriteByte(';')
	}

	if (c & CapMacOverride) != 0 {
		result.WriteString("MAC_OVERRIDE")
		result.WriteByte(';')
	}

	if (c & CapMacAdmin) != 0 {
		result.WriteString("MAC_ADMIN")
		result.WriteByte(';')
	}

	if (c & CapSyslog) != 0 {
		result.WriteString("SYSLOG")
		result.WriteByte(';')
	}

	if (c & CapWakeAlarm) != 0 {
		result.WriteString("WAKE_ALARM")
		result.WriteByte(';')
	}

	if (c & CapBlockSuspend) != 0 {
		result.WriteString("BLOC_SUSPEND")
		result.WriteByte(';')
	}

	return result.String()
}

func CapabilityMaskParse(value string) (CapabilityMask, error) {
	var c CapabilityMask
	caps := strutils.SplitQuoted(value, ';')
	for _, capName := range caps {
		switch capName {
		case "CHOWN":
			c |= CapChown
		case "DAC_OVERRIDE":
			c |= CapDacOverride
		case "DAC_READ_SEARCH":
			c |= CapDacReadSearch
		case "FOWNER":
			c |= CapFowner
		case "FSETID":
			c |= CapFsetid
		case "KILL":
			c |= CapKill
		case "SETGID":
			c |= CapSetgid
		case "SETUID":
			c |= CapSetuid
		case "SETPCAP":
			c |= CapSetpcap
		case "LINUX_IMMUTABLE":
			c |= CapLinuxImmutable
		case "NET_BIND_SERVICE":
			c |= CapNetBindService
		case "NET_BROADCAST":
			c |= CapNetBroadcast
		case "NET_ADMIN":
			c |= CapNetAdmin
		case "NET_RAW":
			c |= CapNetRaw
		case "IPC_LOCK":
			c |= CapIpcLock
		case "IPC_OWNER":
			c |= CapIpcOwner
		case "SYS_MODULE":
			c |= CapSysModule
		case "SYS_RAWIO":
			c |= CapSysRawio
		case "SYS_CHROOT":
			c |= CapSysChroot
		case "SYS_PTRACE":
			c |= CapSysPtrace
		case "SYS_PACCT":
			c |= CapSysPacct
		case "SYS_ADMIN":
			c |= CapSysAdmin
		case "SYS_BOOT":
			c |= CapSysBoot
		case "SYS_NICE":
			c |= CapSysNice
		case "SYS_RESOURCE":
			c |= CapSysResource
		case "SYS_TIME":
			c |= CapSysTime
		case "SYS_TTY_CONFIG":
			c |= CapSysTtyConfig
		case "MKNOD":
			c |= CapMknod
		case "LEASE":
			c |= CapLease
		case "AUDIT_READ":
			c |= CapAuditRead
		case "AUDIT_WRITE":
			c |= CapAuditWrite
		case "AUDIT_CONTROL":
			c |= CapAuditControl
		case "SETFCAP":
			c |= CapSetfcap
		case "MAC_OVERRIDE":
			c |= CapMacOverride
		case "MAC_ADMIN":
			c |= CapMacAdmin
		case "SYSLOG":
			c |= CapSyslog
		case "WAKE_ALARM":
			c |= CapWakeAlarm
		case "BLOC_SUSPEND":
			c |= CapBlockSuspend
		default:
			return 0, fmt.Errorf("unknown capability: %s", capName)
		}
	}
	return c, nil
}
