package porto

import "fmt"

type PortoAccess int

const (
	PortoAccessUnknown PortoAccess = iota
	// no access
	PortoAccessNone
	// read-only access, show only sub-containers
	PortoAccessReadIsolate
	// read-only access
	PortoAccessReadOnly
	// read-write access, show only sub-containers
	PortoAccessIsolate
	// write-access to sub-containers
	PortoAccessChildOnly
	// full access (default)
	PortoAccessFull
)

func (s PortoAccess) String() string {
	switch s {
	case PortoAccessNone:
		return "none"
	case PortoAccessReadIsolate:
		return "read-isolate"
	case PortoAccessReadOnly:
		return "read-only"
	case PortoAccessIsolate:
		return "isolate"
	case PortoAccessChildOnly:
		return "child-only"
	case PortoAccessFull:
		return "full"
	default:
		return "unknown"
	}
}

func PortoAccessParse(value string) (PortoAccess, error) {
	switch value {
	case "false", "none":
		return PortoAccessNone, nil
	case "read-isolate":
		return PortoAccessReadIsolate, nil
	case "read-only":
		return PortoAccessReadOnly, nil
	case "isolate":
		return PortoAccessIsolate, nil
	case "child-only":
		return PortoAccessChildOnly, nil
	case "full", "true":
		return PortoAccessFull, nil
	default:
		return PortoAccessUnknown, fmt.Errorf("unknown virt_mode: %s", value)
	}
}
