package porto

import "fmt"

type State int

const (
	StateUnknown State = iota
	// stopped - initial state (1)
	StateStopped
	// starting - start in progress (16)
	StateStarting
	// running - command execution in progress (2)
	StateRunning
	// stopping - stop in progress (32)
	StateStopping
	// paused - frozen, consumes memory but no cpu (128)
	StatePaused
	// dead - execution complete (4)
	StateDead
	// meta - running container without command (8)
	StateMeta
	// respawning - dead and will be started again (64)
	StateRespawning
)

func (s State) String() string {
	switch s {
	case StateStopped:
		return "stopped"
	case StateStarting:
		return "starting"
	case StateRunning:
		return "running"
	case StateStopping:
		return "stopping"
	case StatePaused:
		return "paused"
	case StateDead:
		return "dead"
	case StateMeta:
		return "meta"
	case StateRespawning:
		return "respawning"
	default:
		return "unknown"
	}
}

func StateParse(name string) (State, error) {
	switch name {
	case "stopped":
		return StateStopped, nil
	case "starting":
		return StateStarting, nil
	case "running":
		return StateRunning, nil
	case "stopping":
		return StateStopping, nil
	case "paused":
		return StatePaused, nil
	case "dead":
		return StateDead, nil
	case "meta":
		return StateMeta, nil
	case "respawning":
		return StateRespawning, nil
	default:
		return StateUnknown, fmt.Errorf("unknown state: %s", name)
	}
}
