package prompt

import (
	"bytes"
	"fmt"
)

type (
	AskOptions struct {
		// Query to ask
		Query string

		// Default value
		Default string

		// MaskDefault hides default value
		MaskDefault bool

		// ValidateFunc is function to do extra validation of user prompt
		ValidateFunc ValidateFunc
	}
)

func (o *AskOptions) validate(input string) error {
	if o.ValidateFunc != nil {
		return o.ValidateFunc(input)
	}

	return nil
}

// Ask asks the user for prompt using the given query
//
// If the user sends SIGINT (Ctrl+C) while reading prompt, it catches
// it and return it as a error.
func (i *UI) Ask(opts AskOptions) (result string, err error) {
	for {
		var buf bytes.Buffer
		_, err = fmt.Fprint(i.writer, opts.Query)
		if err != nil {
			return
		}

		if opts.Default != "" {
			defaultVal := opts.Default

			if opts.MaskDefault {
				defaultVal = maskString(defaultVal)
			}

			buf.WriteString(fmt.Sprintf(" (%s)", defaultVal))
		}

		// Display the instruction to user and ask to prompt.
		buf.WriteString(": ")
		_, err = fmt.Fprint(i.writer, buf.String())
		if err != nil {
			return
		}

		// Read user prompt from UI.Reader.
		result, err = i.read()
		if err != nil {
			return
		}

		// line is empty but default is provided returns it
		if result == "" && opts.Default != "" {
			result = opts.Default
		}

		if result == "" {
			_, err = fmt.Fprint(i.writer, "prompt must not be empty.\n\n")
			if err != nil {
				return
			}
			continue
		}

		if err = opts.validate(result); err != nil {
			_, err = fmt.Fprintf(i.writer, "Failed to validate prompt string: %s\n\n", err)
			if err != nil {
				return
			}
			continue
		}

		break
	}

	return
}
