package pypipkg

import (
	"errors"
	"net/url"
	"path"

	"a.yandex-team.ru/security/libs/go/pypi"
)

const (
	ExtTarBz2 = ".tar.bz2"
	ExtTarGz  = ".tar.gz"
	ExtTar    = ".tar"
	ExtGz     = ".gz"
	ExtBz2    = ".bz2"
	ExtTgz    = ".tgz"
	ExtTbz    = ".tbz"
	ExtTbz2   = ".tbz2"
	ExtZip    = ".zip"
	ExtWhl    = ".whl"
	ExtEgg    = ".egg"
)

var ErrUnknownPkgType = errors.New("unknown pkg format")

func NewPkgReleaseFromURL(pkgName string, downloadURL *url.URL) (release pypi.Release, resultErr error) {
	packageFilename := path.Base(downloadURL.Path)
	release.Type, release.Ext, resultErr = SplitPkgName(packageFilename)
	if resultErr != nil {
		return
	}

	pkgNameVer := packageFilename[:len(packageFilename)-len(release.Ext)]
	switch release.Type {
	case pypi.PackageTypeWheel:
		_, release.Version, resultErr = ParseWheelPath(pkgNameVer)
	case pypi.PackageTypeSrc:
		_, release.Version, resultErr = ParseSourcePath(pkgNameVer, pkgName)
	case pypi.PackageTypeSrcEgg:
		_, release.Version, resultErr = ParseEggPath(pkgNameVer, pkgName)
	default:
		return
	}

	if resultErr != nil {
		return
	}

	release.DownloadURL = downloadURL.String()
	return
}

func SplitPkgName(pkgFilename string) (pkgType pypi.PackageType, ext string, err error) {
	ext = path.Ext(pkgFilename)
	switch ext {
	case ExtGz:
		ext = path.Ext(pkgFilename[:len(pkgFilename)-len(ExtGz)]) + ExtGz
	case ExtBz2:
		ext = path.Ext(pkgFilename[:len(pkgFilename)-len(ExtBz2)]) + ExtBz2
	}

	switch ext {
	case ExtWhl:
		pkgType = pypi.PackageTypeWheel
	case ExtTarGz, ExtTarBz2, ExtTar, ExtZip, ExtTgz, ExtTbz, ExtTbz2:
		// See source_extensions at distlib: https://clck.ru/Fjyf3
		pkgType = pypi.PackageTypeSrc
	case ExtEgg:
		pkgType = pypi.PackageTypeSrcEgg
	default:
		err = ErrUnknownPkgType
	}

	return
}
