package pypipkg

import (
	"errors"
	"fmt"
	"net/url"
	"regexp"
	"strings"
)

var (
	ErrFormat = errors.New("invalid format")

	nameReplaceRe     = regexp.MustCompile(`[-_.]+`)
	sourceFileRe      = regexp.MustCompile(`(?i)^([a-z0-9_]+(?:[.-][a-z_][a-z0-9_]*)*)-([a-z0-9_.+-]+)`)
	wheelFileRe       = regexp.MustCompile(`^(.+?)-(.*?)-`)
	sourcePythonVerRe = regexp.MustCompile(`-py\d\.?\d?`)
)

// Normalize a python package name a la PEP 503
// Reference: https://clck.ru/Fjyfd
func NormalizeName(name string) string {
	return strings.ToLower(nameReplaceRe.ReplaceAllString(name, "-"))
}

// Extract name and version from a wheel pkg filename (no extension)
// Reference: https://clck.ru/Fjyfy
func ParseWheelPath(wheelFilename string) (name string, version string, resultErr error) {
	parsed := wheelFileRe.FindStringSubmatch(wheelFilename)
	if parsed == nil {
		resultErr = ErrFormat
		return
	}

	name = parsed[1]
	version = parsed[2]
	return
}

// Extract name and version from a source pkg filename (no extension)
// Reference: https://clck.ru/FjygE
func ParseSourcePath(sourceFilename string, sourceName string) (name string, version string, resultErr error) {
	filename, err := url.PathUnescape(sourceFilename)
	if err != nil {
		resultErr = fmt.Errorf("unescape path failed: %w", resultErr)
		return
	}

	filename = strings.Replace(filename, " ", "-", -1)
	if indx := sourcePythonVerRe.FindStringIndex(filename); indx != nil {
		filename = filename[0:indx[0]]
	}

	if sourceName != "" && len(filename) > len(sourceName)+1 && strings.HasPrefix(filename, sourceName) {
		name = sourceName
		version = filename[len(sourceName)+1:]
	} else if raw := sourceFileRe.FindStringSubmatch(filename); raw != nil {
		name = raw[1]
		version = raw[2]
		return
	} else {
		resultErr = ErrFormat
	}

	return
}

func ParseEggPath(sourceFilename string, sourceName string) (name string, version string, resultErr error) {
	return ParseSourcePath(sourceFilename, sourceName)
}
