package pypilocalshop

import (
	"fmt"
	"io"
	"net/http/cookiejar"
	"net/url"
	"strings"
	"time"

	"golang.org/x/net/html"

	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

const (
	shopPkgTypeMirror = "pypi package"
	shopPkgTypeLocal  = "local package"
)

type (
	Options struct {
		LocalOnly bool
		Name      string
		BaseURL   string
		Login     string
		Password  string
	}

	Client struct {
		name       string
		baseURL    *url.URL
		httpClient *yahttp.Client
		err        error
		current    int
		total      int
		rows       []indexRow
		localOnly  bool
		parsed     bool
	}

	indexRow struct {
		PkgURL  string
		PkgType string

		rowStarted bool
		tdLink     bool
		tdType     bool
	}
)

func New(opts Options) (client *Client, resultErr error) {
	client = new(Client)
	client.name = opts.Name
	client.localOnly = opts.LocalOnly
	client.baseURL, resultErr = url.Parse(opts.BaseURL)
	if resultErr != nil {
		resultErr = fmt.Errorf("failed to parse pypi host: %w", resultErr)
		return
	}
	client.baseURL.Path = strings.TrimRight(client.baseURL.Path, "/") + "/dashboard/repositories/default/"

	cookieJar, _ := cookiejar.New(nil)
	// TODO(buglloc): custom client
	client.httpClient = yahttp.NewClient(yahttp.Config{
		RedirectPolicy: yahttp.RedirectFollowSameOrigin,
		DialTimeout:    4 * time.Second,
		Timeout:        320 * time.Second,
	})
	client.httpClient.Jar = cookieJar

	if resultErr = client.auth(opts.Login, opts.Password); resultErr != nil {
		return
	}
	return
}

func (p *Client) Name() string {
	return p.name
}

func (p *Client) Error() error {
	return p.err
}

func (p *Client) Next() bool {
	if !p.parsed {
		p.err = p.parse()
		p.current = -1
		p.total = len(p.rows)
		p.parsed = true
	}

	if p.err != nil {
		return false
	}

	if p.current >= p.total-1 {
		return false
	}

	p.current++
	return true
}

func (p *Client) Package() (pypi.Package, error) {
	row := p.rows[p.current]
	relativePkgURL, err := url.Parse(row.PkgURL)
	if err != nil {
		return nil, fmt.Errorf("failed to parse pkg url: %w", err)
	}

	pkgURL := p.baseURL.ResolveReference(relativePkgURL)
	name := strings.TrimRight(pkgURL.Path, "/")
	if idx := strings.LastIndexByte(name, '/'); idx != -1 {
		name = name[idx+1:]
	}

	return &Pkg{
		name:       name,
		pkgURL:     pkgURL.String(),
		httpClient: p.httpClient,
	}, nil
}

func (p *Client) FindPackage(name string) (pypi.Package, error) {
	pkgURL := *p.baseURL
	pkgURL.Path = fmt.Sprintf("/dashboard/repositories/default/packages/%s/", url.PathEscape(name))

	return &Pkg{
		name:       name,
		pkgURL:     pkgURL.String(),
		httpClient: p.httpClient,
	}, nil
}

func (p *Client) parse() error {
	resp, err := p.httpClient.Get(p.baseURL.String())
	if err != nil {
		return err
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return fmt.Errorf("pypi returns non 200 status code: %d", resp.StatusCode)
	}

	indexScanner := html.NewTokenizer(resp.Body)
	for {
		tokenType := indexScanner.Next()
		switch tokenType {
		case html.ErrorToken:
			if err := indexScanner.Err(); err != io.EOF {
				return err
			}
			return nil
		case html.StartTagToken:
			token := indexScanner.Token()
			// wait tbody
			if token.Data != tagTBody {
				continue
			}

			for {
				row, err := parseIndexRow(indexScanner)
				if err != nil {
					if err != io.EOF {
						return err
					}
					return nil
				}

				if row.PkgURL == "" {
					// skip w/o pkg url
					continue
				}

				switch row.PkgType {
				case shopPkgTypeMirror:
					if p.localOnly {
						continue
					}
				case shopPkgTypeLocal:
					// ok
				default:
					return fmt.Errorf("unknown pkg type: %s", row.PkgType)
				}

				p.rows = append(p.rows, row)
			}
		}
	}
}

func parseIndexRow(indexScanner *html.Tokenizer) (row indexRow, err error) {
	for {
		tokenType := indexScanner.Next()
		switch tokenType {
		case html.ErrorToken:
			// done!
			err = indexScanner.Err()
			return
		case html.StartTagToken:
			token := indexScanner.Token()
			switch token.Data {
			case tagTr:
				row.rowStarted = true
			case tagTd:
				switch {
				case !row.rowStarted:
					break
				case !row.tdLink:
					row.tdLink = true
				case !row.tdType:
					row.tdType = true
					tokenType := indexScanner.Next()
					if tokenType != html.TextToken {
						err = fmt.Errorf("expected text token, got: %s", tokenType.String())
						return
					}
					row.PkgType = strings.TrimSpace(indexScanner.Token().Data)
				}
			case tagA:
				if !row.tdLink {
					continue
				}

				for _, a := range token.Attr {
					if a.Key == attrHref {
						row.PkgURL = a.Val
						break
					}
				}
			}
		case html.EndTagToken:
			if indexScanner.Token().Data == tagTr {
				// done! yeah!
				return
			}
		}
	}
}

func (p *Client) auth(login, password string) error {
	resp, err := p.httpClient.PostForm("https://passport.yandex-team.ru/passport?mode=auth", url.Values{
		"login":  {login},
		"passwd": {password},
	})
	if err != nil {
		return fmt.Errorf("failed to auth: %w", err)
	}

	yahttp.GracefulClose(resp.Body)
	return nil
}
