package pypilocalshop

import (
	"fmt"
	"io"
	"net/url"
	"strings"

	"golang.org/x/net/html"

	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type Pkg struct {
	name       string
	normName   string
	pkgURL     string
	releases   map[string][]pypi.Release
	httpClient *yahttp.Client
}

type pkgFile struct {
	url      string
	filename string
}

type pkgVersion struct {
	version string
	files   []pkgFile
}

func (p *Pkg) Name() string {
	return p.name
}

func (p *Pkg) NormName() string {
	if p.normName != "" {
		return p.normName
	}

	p.normName = pypipkg.NormalizeName(p.name)
	return p.normName
}

func (p *Pkg) Releases() pypi.Releases {
	return p.releases
}

func (p *Pkg) Resolve() error {
	if p.releases != nil {
		return nil
	}

	simplelog.Debug("resolve pkg", "url", p.pkgURL)
	resp, err := p.httpClient.Get(p.pkgURL)
	if err != nil {
		return err
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return fmt.Errorf("pypi returns non 200 status code: %d", resp.StatusCode)
	}

	p.releases = make(pypi.Releases)
	baseURL, _ := url.Parse(p.pkgURL)
	htmlScanner := html.NewTokenizer(resp.Body)
	tBodyCount := 0
	for {
		tokenType := htmlScanner.Next()
		switch tokenType {
		case html.ErrorToken:
			// Yeah! we're done!
			if err := htmlScanner.Err(); err != io.EOF {
				return err
			}
			return nil
		case html.StartTagToken:
			token := htmlScanner.Token()

			// Wait send tbody (look at HTML source, please)
			if token.Data != tagTBody {
				continue
			}

			tBodyCount++
			if tBodyCount != 2 {
				continue
			}

			versions, err := parseVersions(htmlScanner)
			if err != nil {
				return fmt.Errorf("failed to list pkg versions: %w", err)
			}

			for _, parsedVersion := range versions {
				for _, file := range parsedVersion.files {
					fileURL, err := url.Parse(file.url)
					if err != nil {
						simplelog.Error("failed to parse package url", "url", file.url, "err", err)
						continue
					}

					if file.filename == "" {
						simplelog.Error("empty filename", "pkg_url", p.pkgURL)
						continue
					}

					pkgType, ext, err := pypipkg.SplitPkgName(file.filename)
					if err != nil {
						simplelog.Error("failed to parse pkg filename",
							"pkg_filename", file.filename,
							"pkg_url", p.pkgURL,
						)
						continue
					}

					p.releases[parsedVersion.version] = append(p.releases[parsedVersion.version], pypi.Release{
						Type:        pkgType,
						Ext:         ext,
						Version:     parsedVersion.version,
						DownloadURL: baseURL.ResolveReference(fileURL).String(),
					})
				}
			}
		}
	}
}

func parseVersions(htmlScanner *html.Tokenizer) (versions []pkgVersion, resultErr error) {
	versionStarted := false
	var version pkgVersion
	for {
		tokenType := htmlScanner.Next()
		switch tokenType {
		case html.ErrorToken:
			resultErr = htmlScanner.Err()
			return
		case html.StartTagToken:
			token := htmlScanner.Token()
			switch token.Data {
			case tagTh:
				if versionStarted {
					versions = append(versions, version)
				}

				versionStarted = true
				version = pkgVersion{}
				tokenType := htmlScanner.Next()
				if tokenType != html.TextToken {
					resultErr = fmt.Errorf("expected text token, got: %s", tokenType.String())
					return
				}
				version.version = strings.TrimSpace(htmlScanner.Token().Data)
			case tagA:
				if !versionStarted {
					continue
				}

				file := pkgFile{}
				for _, a := range token.Attr {
					if a.Key == attrHref {
						file.url = a.Val
						break
					}
				}

				tokenType := htmlScanner.Next()
				if tokenType != html.TextToken {
					resultErr = fmt.Errorf("expected text token, got: %s", tokenType.String())
					return
				}
				file.filename = strings.TrimSpace(htmlScanner.Token().Data)
				version.files = append(version.files, file)
			}
		case html.EndTagToken:
			if htmlScanner.Token().Data == tagTBody {
				if versionStarted {
					versions = append(versions, version)
				}

				return
			}
		}
	}
}
