package pypipypi

import (
	"encoding/xml"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type (
	Options struct {
		HTTPClient *yahttp.Client
		Name       string
		BaseURL    string
	}

	Client struct {
		name           string
		pkgURLTmpl     string
		err            error
		baseURL        string
		current        int
		total          int
		parsed         bool
		remotePackages []xmlPackage
		httpClient     *yahttp.Client
	}

	xmlPackage struct {
		XMLName xml.Name `xml:"value"`
		Name    string   `xml:"string"`
	}

	xmlResponse struct {
		XMLName  string       `xml:"methodResponse"`
		Packages []xmlPackage `xml:"params>param>value>array>data>value"`
	}
)

func New(opts Options) (client *Client, resultErr error) {
	client = new(Client)
	client.name = opts.Name
	if opts.HTTPClient != nil {
		client.httpClient = opts.HTTPClient
	} else {
		client.httpClient = yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectFollowSameOrigin,
			DialTimeout:    4 * time.Second,
			Timeout:        20 * time.Second,
		})
	}

	baseURL, err := url.Parse(opts.BaseURL)
	if err != nil {
		resultErr = fmt.Errorf("failed to parse pypi host: %w", err)
		return
	}
	baseURL.Path = strings.TrimRight(baseURL.Path, "/") + "/pypi/"

	client.pkgURLTmpl = fmt.Sprintf(
		"%s://%s%s%%s/json",
		baseURL.Scheme, baseURL.Host, baseURL.Path,
	)
	client.baseURL = baseURL.String()

	return
}

func (p *Client) Name() string {
	return p.name
}

func (p *Client) Error() error {
	return p.err
}

func (p *Client) Next() bool {
	if !p.parsed {
		p.err = p.parse()
		p.current = -1
		p.total = len(p.remotePackages)
		p.parsed = true
	}

	if p.err != nil {
		return false
	}

	if p.current >= p.total-1 {
		return false
	}

	p.current++
	return true
}

func (p *Client) Package() (pypi.Package, error) {
	return p.FindPackage(p.remotePackages[p.current].Name)
}

func (p *Client) FindPackage(name string) (pypi.Package, error) {
	return &Pkg{
		name:       name,
		infoURL:    fmt.Sprintf(p.pkgURLTmpl, name),
		httpClient: p.httpClient,
	}, nil
}

func (p *Client) parse() error {
	body := strings.NewReader("<methodCall><methodName>list_packages</methodName><params></params></methodCall>")
	req, err := http.NewRequest("POST", p.baseURL, body)
	if err != nil {
		return fmt.Errorf("failed to create pypi request: %w", err)
	}

	req.Header.Set("Content-Type", "text/xml")
	resp, err := p.httpClient.Do(req)
	if err != nil {
		return fmt.Errorf("failed to make pypi request: %w", err)
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return fmt.Errorf("pypi returns non 200 status code: %d", resp.StatusCode)
	}

	var methodResponse xmlResponse
	if err = xml.NewDecoder(resp.Body).Decode(&methodResponse); err != nil {
		return fmt.Errorf("failed to decode pypi response: %w", err)
	}

	p.remotePackages = methodResponse.Packages
	return nil
}
