package pypipypi

import (
	"encoding/json"
	"fmt"

	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type Pkg struct {
	name       string
	normName   string
	infoURL    string
	releases   map[string][]pypi.Release
	httpClient *yahttp.Client
}

type pkgInfo struct {
	Info struct {
		Name string `json:"name"`
	} `json:"info"`

	Releases map[string][]struct {
		Filename    string `json:"filename"`
		PackageType string `json:"packagetype"`
		Size        int    `json:"size"`
		DownloadURL string `json:"url"`
	} `json:"releases"`
}

func (p *Pkg) Name() string {
	return p.name
}

func (p *Pkg) NormName() string {
	if p.normName != "" {
		return p.normName
	}

	p.normName = pypipkg.NormalizeName(p.name)
	return p.normName
}

func (p *Pkg) Releases() pypi.Releases {
	return p.releases
}

func (p *Pkg) Resolve() error {
	if p.releases != nil {
		return nil
	}

	simplelog.Debug("resolve pkg", "url", p.infoURL)
	resp, err := p.httpClient.Get(p.infoURL)
	if err != nil {
		return err
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return fmt.Errorf("pypi returns non 200 status code: %d", resp.StatusCode)
	}

	var info pkgInfo
	err = json.NewDecoder(resp.Body).Decode(&info)
	if err != nil {
		return fmt.Errorf("failed to decode pypi response: %w", err)
	}

	p.name = info.Info.Name
	p.releases = make(pypi.Releases, len(info.Releases))
	for version, releases := range info.Releases {
		for _, release := range releases {
			if release.Filename == "" {
				simplelog.Error("empty filename", "pkg_url", p.infoURL)
				continue
			}

			pkgType, ext, err := pypipkg.SplitPkgName(release.Filename)
			if err != nil {
				simplelog.Error("failed to parse pkg filename",
					"pkg_filename", release.Filename,
					"pkg_url", p.infoURL,
				)
				continue
			}

			p.releases[version] = append(p.releases[version], pypi.Release{
				Type:        pkgType,
				Ext:         ext,
				Version:     version,
				DownloadURL: release.DownloadURL,
				Size:        release.Size,
			})
		}
	}

	return nil
}
