package pypisimple

import (
	"bytes"
	"fmt"
	"io"
	"net/url"
	"strings"
	"time"

	"golang.org/x/net/html"

	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type (
	Options struct {
		HTTPClient *yahttp.Client
		Name       string
		BaseURL    string
	}

	Client struct {
		name         string
		baseURL      *url.URL
		httpClient   *yahttp.Client
		err          error
		parsed       bool
		indexScanner *html.Tokenizer
		currentURL   string
	}
)

func New(opts Options) (client *Client, resultErr error) {
	client = new(Client)
	client.name = opts.Name
	client.baseURL, resultErr = url.Parse(opts.BaseURL)
	if resultErr != nil {
		resultErr = fmt.Errorf("failed to parse pypi host: %w", resultErr)
		return
	}
	client.baseURL.Path = strings.TrimRight(client.baseURL.Path, "/") + "/simple/"

	if opts.HTTPClient != nil {
		client.httpClient = opts.HTTPClient
	} else {
		client.httpClient = yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectFollowSameOrigin,
			DialTimeout:    4 * time.Second,
			Timeout:        20 * time.Second,
		})
	}

	return
}

func (p *Client) Name() string {
	return p.name
}

func (p *Client) Error() error {
	return p.err
}

func (p *Client) Next() bool {
	if !p.parsed {
		p.err = p.parse()
		p.parsed = true
	}

	if p.err != nil {
		return false
	}

	for {
		tokenType := p.indexScanner.Next()
		switch tokenType {
		case html.ErrorToken:
			if err := p.indexScanner.Err(); err != io.EOF {
				p.err = err
			}
			return false
		case html.StartTagToken:
			token := p.indexScanner.Token()
			// skip non <a> tags
			if token.Data != "a" {
				continue
			}

			var packageURL string
			for _, a := range token.Attr {
				if a.Key == "href" {
					packageURL = a.Val
					break
				}
			}

			if packageURL == "" {
				// skip empty <a> tags
				continue
			}

			p.currentURL = packageURL
			return true
		}
	}
}

func (p *Client) Package() (pypi.Package, error) {
	releasesURL, err := p.ReleasesURL()
	if err != nil {
		return nil, err
	}

	name := strings.TrimRight(releasesURL.Path, "/")
	if idx := strings.LastIndexByte(name, '/'); idx != -1 {
		name = name[idx+1:]
	}

	return &Pkg{
		name:        name,
		releasesURL: releasesURL.String(),
		httpClient:  p.httpClient,
	}, nil
}

func (p *Client) FindPackage(name string) (pypi.Package, error) {
	releasesURL := *p.baseURL
	if releasesURL.Host == "pypi.yandex-team.ru" {
		// Special hack for Yandex internal PyPi
		releasesURL.Path = fmt.Sprintf("/repo/default/%s/", url.PathEscape(pypipkg.NormalizeName(name)))
	} else {
		releasesURL.Path = fmt.Sprintf("/simple/%s/", url.PathEscape(pypipkg.NormalizeName(name)))
	}

	return &Pkg{
		name:        name,
		releasesURL: releasesURL.String(),
		httpClient:  p.httpClient,
	}, nil
}

func (p *Client) ReleasesURL() (*url.URL, error) {
	relativeURL, err := url.Parse(p.currentURL)
	if err != nil {
		return nil, fmt.Errorf("failed to parse package url: %w", err)
	}

	releasesURL := p.baseURL.ResolveReference(relativeURL)
	return releasesURL, nil
}

func (p *Client) PackageName() (name string, resultErr error) {
	releasesURL, err := p.ReleasesURL()
	if err != nil {
		resultErr = err
		return
	}

	name = strings.TrimRight(releasesURL.Path, "/")
	if idx := strings.LastIndexByte(name, '/'); idx != -1 {
		name = name[idx+1:]
	}
	return
}

func (p *Client) parse() error {
	resp, err := p.httpClient.Get(p.baseURL.String())
	if err != nil {
		return err
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return fmt.Errorf("pypi returns non 200 status code: %d", resp.StatusCode)
	}

	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("failed to read pypi response: %w", err)
	}

	p.indexScanner = html.NewTokenizer(bytes.NewReader(body))
	return nil
}
