package pypisimple

import (
	"fmt"
	"io"
	"net/url"

	"golang.org/x/net/html"

	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type Pkg struct {
	name        string
	normName    string
	releasesURL string
	releases    map[string][]pypi.Release
	httpClient  *yahttp.Client
}

func (p *Pkg) Name() string {
	return p.name
}

func (p *Pkg) NormName() string {
	if p.normName != "" {
		return p.normName
	}

	p.normName = pypipkg.NormalizeName(p.name)
	return p.normName
}

func (p *Pkg) Releases() pypi.Releases {
	return p.releases
}

func (p *Pkg) Resolve() error {
	if p.releases != nil {
		return nil
	}

	simplelog.Debug("resolve pkg", "url", p.releasesURL)
	resp, err := p.httpClient.Get(p.releasesURL)
	if err != nil {
		return err
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return fmt.Errorf("pypi returns non 200 status code: %d", resp.StatusCode)
	}

	p.releases = make(pypi.Releases)
	baseURL, _ := url.Parse(p.releasesURL)
	htmlScanner := html.NewTokenizer(resp.Body)
	for {
		tokenType := htmlScanner.Next()
		switch tokenType {
		case html.ErrorToken:
			// Yeah! we're done!
			if err := htmlScanner.Err(); err != io.EOF {
				return err
			}
			return nil
		case html.StartTagToken:
			token := htmlScanner.Token()

			// Check if the token is an <a> tag
			if token.Data != "a" {
				continue
			}

			var rawReleasesURL string
			for _, a := range token.Attr {
				if a.Key == "href" {
					rawReleasesURL = a.Val
					break
				}
				// not all pypis have rel attr
				//else if a.Key == "rel" {
				//	isPackage = a.Val == "package" || a.Val == "internal"
				//}
			}

			if rawReleasesURL == "" {
				continue
			}

			packageURL, err := url.Parse(rawReleasesURL)
			if err != nil {
				simplelog.Debug("failed to parse package url", "url", rawReleasesURL, "err", err)
				continue
			}

			release, err := pypipkg.NewPkgReleaseFromURL(p.name, baseURL.ResolveReference(packageURL))
			if err != nil {
				simplelog.Debug("failed to parse pkg release",
					"url", rawReleasesURL,
					"err", err,
				)
				continue
			}

			p.releases[release.Version] = append(p.releases[release.Version], release)
		}
	}
}
