package retry

import (
	"context"
	"time"

	"a.yandex-team.ru/security/libs/go/retry/backoff"
)

const (
	DefaultAttempts = 3
)

type (
	// Retrier interface is used for any retrier implementation
	Retrier interface {
		// Try calls the work function until it does not return error
		// or retry break condition happens.
		//
		// Returns last error if any
		Try(ctx context.Context, work TryFunc) error

		// TryNotify calls the work function until it does not return error
		// or retry break condition happens.
		// Notify is called with the error and wait duration
		// for each failed attempt before sleep.
		//
		// Returns last error if any
		TryNotify(ctx context.Context, work TryFunc, notify NotifyFunc) error
	}

	ConditionFunc func(err error, attempt int) bool

	TryFunc func(ctx context.Context) error

	NotifyFunc func(err error, delay time.Duration)

	RealRetry struct {
		Options
	}

	NopRetry struct {
	}
)

func New(opts ...Option) Retrier {
	options := Options{
		Attempts: DefaultAttempts,
		BackOff:  backoff.DefaultExponential,
	}

	for _, opt := range opts {
		opt(&options)
	}

	if options.Attempts == 1 {
		return NopRetry{}
	}

	return RealRetry{
		Options: options,
	}
}

func (r RealRetry) Try(ctx context.Context, work TryFunc) (err error) {
	return r.TryNotify(ctx, work, nil)
}

func (r RealRetry) TryNotify(ctx context.Context, work TryFunc, notify NotifyFunc) (err error) {
	max := r.Attempts - 1
	for attempt := 0; ; attempt++ {
		err = work(ctx)
		if err == nil {
			return nil
		}

		if max >= 0 && attempt == max {
			return err
		}

		if r.Condition != nil && !r.Condition(err, attempt) {
			break
		}

		delay := r.BackOff.Delay(err, attempt)
		if notify != nil {
			notify(err, delay)
		}

		select {
		case <-time.After(delay):
			// pass
		case <-ctx.Done():
			return ctx.Err()
		}
	}

	return err
}

func (r NopRetry) Try(ctx context.Context, work TryFunc) error {
	return work(ctx)
}

func (r NopRetry) TryNotify(ctx context.Context, work TryFunc, notify NotifyFunc) error {
	return work(ctx)
}
