package mocktools

import (
	"fmt"
	"io"
	"os"
	"runtime"

	"github.com/klauspost/compress/zstd"

	"a.yandex-team.ru/security/libs/go/hashreader"
	"a.yandex-team.ru/security/libs/go/sectools"
)

type Option func(*Handler) error

func WithTool(name, channel, version, binaryPath string) Option {
	return func(handler *Handler) error {
		toolReader, err := os.Open(binaryPath)
		if err != nil {
			return err
		}
		defer func() { _ = toolReader.Close() }()

		encR, encW := io.Pipe()
		defer func() {
			_ = encR.Close()
		}()

		encoder, err := zstd.NewWriter(encW)
		if err != nil {
			return err
		}

		go func() {
			_, _ = io.Copy(encoder, toolReader)
			_ = encoder.Close()
			_ = encW.Close()
		}()

		hashedR, err := hashreader.NewHashReader(encR)
		if err != nil {
			return err
		}

		binaryBytes, err := io.ReadAll(hashedR)
		if err != nil {
			return err
		}

		manifest := sectools.Manifest{
			Version: version,
			Binaries: sectools.BinariesMap{
				runtime.GOOS: {
					runtime.GOARCH: {
						URL:     fmt.Sprintf("/%s/%s/%s_%s.zst", name, version, name, version),
						FastURL: fmt.Sprintf("/%s/%s/%s_%s.zst", name, version, name, version),
						DumbURL: fmt.Sprintf("/%s/%s/%s_%s.zst", name, version, name, version),
						Size:    hashedR.Size(),
						Hash:    hashedR.Hash(),
					},
				},
			},
		}

		handler.tools = append(
			handler.tools,
			Tool{
				Name:     name,
				Channel:  channel,
				Binary:   binaryBytes,
				Manifest: manifest,
			},
			Tool{
				Name:     name,
				Channel:  version,
				Binary:   binaryBytes,
				Manifest: manifest,
			},
		)
		return nil
	}
}

func WithManifest(name string, manifest sectools.Manifest) Option {
	return func(handler *Handler) error {
		handler.tools = append(handler.tools, Tool{
			Name:     name,
			Manifest: manifest,
		})
		return nil
	}
}
