package mocktools

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"

	"github.com/go-chi/chi/v5"
)

type Handler struct {
	tools []Tool
}

func NewSrv(opts ...Option) (*httptest.Server, error) {
	handler := &Handler{}
	for _, opt := range opts {
		if err := opt(handler); err != nil {
			return nil, err
		}
	}

	r := chi.NewRouter()
	r.Get("/{tool}/{version}/manifest.json", handler.manifest)
	r.Get("/{tool}/{version}/{filename}", handler.file)
	return httptest.NewServer(r), nil
}

func (s *Handler) manifest(w http.ResponseWriter, r *http.Request) {
	tool := chi.URLParam(r, "tool")
	version := chi.URLParam(r, "version")
	for _, t := range s.tools {
		if t.Name != tool {
			continue
		}

		if t.Channel != version {
			continue
		}

		_ = json.NewEncoder(w).Encode(t.Manifest)
		return
	}

	http.Error(w, fmt.Sprintf("tool %s@%s was not found", tool, version), http.StatusNotFound)
}

func (s *Handler) file(w http.ResponseWriter, r *http.Request) {
	tool := chi.URLParam(r, "tool")
	version := chi.URLParam(r, "version")
	filename := chi.URLParam(r, "filename")
	expectedFilename := fmt.Sprintf("%s_%s.zst", tool, version)
	if expectedFilename != chi.URLParam(r, "filename") {
		http.Error(w, fmt.Sprintf("unexpected filename, must be: %s", filename), http.StatusBadRequest)
		return
	}

	for _, t := range s.tools {
		if t.Name != tool {
			continue
		}

		if t.Manifest.Version != version {
			continue
		}

		_, _ = w.Write(t.Binary)
		return
	}
}
