package semver

import (
	"errors"
	"strings"

	"github.com/blang/semver/v4"
)

var (
	// ErrEmptyString is returned when an empty string is passed in for parsing.
	ErrEmptyString = errors.New("version string empty")
)

// Version represents a single semantic version.
type Version struct {
	semver.Version
	Original string
}

// StrictNewVersion parses a given version and returns an instance of Version or
// an error if unable to parse the version. Only parses valid semantic versions.
// Performs checking that can find errors within the version.
// If you want to coerce a version, such as 1 or 1.2, and perse that as the 1.x
// releases of semver provided use the NewSemver() function.
func StrictNewVersion(v string) (*Version, error) {
	if len(v) == 0 {
		return nil, ErrEmptyString
	}

	semVer, err := semver.Parse(v)
	if err != nil {
		return nil, err
	}

	if err = semVer.Validate(); err != nil {
		return nil, err
	}

	return &Version{Version: semVer, Original: v}, nil
}

// NewVersion parses a given version and returns an instance of Version or
// an error if unable to parse the version. If the version is SemVer-ish it
// attempts to convert it to SemVer. If you want  to validate it was a strict
// semantic version at parse time see StrictNewVersion().
func NewVersion(v string) (*Version, error) {
	rawVersion := strings.TrimSpace(v)
	rawVersion = strings.TrimPrefix(rawVersion, "v")

	// Split into major.minor.(patch+pr+meta)
	var suffix string
	s := rawVersion
	if idx := strings.IndexAny(rawVersion, "+-"); idx > -1 {
		suffix = rawVersion[idx:]
		s = rawVersion[:idx]
	}

	parts := strings.SplitN(s, ".", 3)
	if len(parts) < 3 {
		for len(parts) < 3 {
			parts = append(parts, "0")
		}

		rawVersion = strings.Join(parts, ".")
		if suffix != "" {
			rawVersion += suffix
		}
	}

	semVer, err := semver.Parse(rawVersion)
	if err != nil {
		return nil, err
	}

	return &Version{Version: semVer, Original: v}, nil
}

// MustParse parses a given version and panics on error.
func MustParse(v string) *Version {
	sv, err := NewVersion(v)
	if err != nil {
		panic(err)
	}
	return sv
}

func (v *Version) Prerelease() string {
	out := make([]string, len(v.Pre))
	for i, p := range v.Pre {
		out[i] = p.String()
	}
	return strings.Join(out, ".")
}

func (v *Version) Metadata() string {
	return strings.Join(v.Build, ".")
}

// LessThan tests if one version is less than another one.
func (v *Version) LessThan(o *Version) bool {
	return v.Compare(o) < 0
}

// GreaterThan tests if one version is greater than another one.
func (v *Version) GreaterThan(o *Version) bool {
	return v.Compare(o) > 0
}

// Equal tests if two versions are equal to each other.
// Note, versions can be equal with different metadata since metadata
// is not considered part of the comparable version.
func (v *Version) Equal(o *Version) bool {
	return v.Compare(o) == 0
}

// Compare compares this version to another one. It returns -1, 0, or 1 if
// the version smaller, equal, or larger than the other version.
func (v *Version) Compare(o *Version) int {
	return v.Version.Compare(o.Version)
}

func comparePrerelease(v, o []semver.PRVersion) int {
	// Quick comparison if a version has no prerelease versions
	switch {
	case len(v) == 0 && len(o) == 0:
		return 0
	case len(v) == 0 && len(o) > 0:
		return 1
	case len(v) > 0 && len(o) == 0:
		return -1
	}

	i := 0
	for ; i < len(v) && i < len(o); i++ {
		if comp := v[i].Compare(o[i]); comp != 0 {
			return comp
		}
	}

	// If all pr versions are the equal but one has further prversion, this one greater
	switch {
	case i == len(v) && i == len(o):
		return 0
	case i == len(v) && i < len(o):
		return -1
	default:
		return 1
	}
}
