package sfx

import (
	"archive/zip"
	"errors"
	"fmt"
	"io"
	"os"
	"path"
	"path/filepath"
	"strings"
)

// IsZip checks to see if path is already a zip file
func IsZip(path string) bool {
	r, err := zip.OpenReader(path)
	if err != nil {
		return false
	}

	_ = r.Close()
	return true
}

// Zip takes all the files (dirs) and zips them into path
func Zip(path string, dirs []string) (err error) {
	if IsZip(path) {
		return errors.New("already a zip file")
	}

	f, err := os.OpenFile(path, os.O_RDWR|os.O_CREATE|os.O_APPEND, 0644)
	if err != nil {
		return
	}
	defer func() { _ = f.Close() }()

	startoffset, err := f.Seek(0, io.SeekEnd)
	if err != nil {
		return
	}

	w := zip.NewWriter(f)
	w.SetOffset(startoffset)

	for _, dir := range dirs {
		absDir, err := filepath.Abs(dir)
		if err != nil {
			return fmt.Errorf("unable to resolve dir %q: %w", dir, err)
		}
		pathPrefix := absDir + string(filepath.Separator)

		err = filepath.Walk(dir, func(path string, info os.FileInfo, err error) error {
			if err != nil {
				return err
			}

			if path == dir {
				return nil
			}

			fh, err := zip.FileInfoHeader(info)
			if err != nil {
				return fmt.Errorf("create file info for %q: %w", path, err)
			}
			fh.Name = strings.TrimPrefix(path, pathPrefix)

			p, err := w.CreateHeader(fh)
			if err != nil {
				return fmt.Errorf("create zip header for %q: %w", path, err)
			}

			if info.IsDir() {
				return nil
			}

			content, err := os.ReadFile(path)
			if err != nil {
				return err
			}

			_, err = p.Write(content)
			if err != nil {
				return fmt.Errorf("unable to write file %q into archive: %w", path, err)
			}
			return err
		})

		if err != nil {
			return fmt.Errorf("process dir %q: %w", dir, err)
		}
	}

	return w.Close()
}

// Unzip unzips the file zippath and puts it in destination
func Unzip(zippath string, destination string) error {
	r, err := zip.OpenReader(zippath)
	if err != nil {
		return err
	}

	for _, f := range r.File {
		fullname := path.Join(destination, f.Name)
		if f.FileInfo().IsDir() {
			if err := os.MkdirAll(fullname, f.FileInfo().Mode().Perm()); err != nil {
				return err
			}
			continue
		}

		if err := os.MkdirAll(filepath.Dir(fullname), 0755); err != nil {
			return err
		}

		perms := f.FileInfo().Mode().Perm()
		out, err := os.OpenFile(fullname, os.O_CREATE|os.O_RDWR, perms)
		if err != nil {
			return err
		}
		rc, err := f.Open()
		if err != nil {
			return err
		}
		_, err = io.CopyN(out, rc, f.FileInfo().Size())
		if err != nil {
			return err
		}

		_ = rc.Close()
		_ = out.Close()

		mtime := f.FileInfo().ModTime()
		err = os.Chtimes(fullname, mtime, mtime)
		if err != nil {
			return err
		}
	}

	return nil
}

// List Lists all the files in zip file
func List(path string) (list []string, err error) {
	r, err := zip.OpenReader(path)
	if err != nil {
		return
	}

	defer func() { _ = r.Close() }()

	for _, f := range r.File {
		if f.FileInfo().IsDir() {
			continue
		}

		list = append(list, f.Name)
	}
	return
}
