package main

import (
	"context"
	"fmt"
	"os"

	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/libs/go/signer"
)

func fatalf(msg string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "signer:"+msg+"\n", a...)
	os.Exit(1)
}

func main() {
	var src, dst, app, comment string
	pflag.StringVar(&src, "src", "", "source file")
	pflag.StringVar(&dst, "dst", "", "destination file")
	pflag.StringVar(&app, "app", "", "app")
	pflag.StringVar(&comment, "comment", "", "comment")
	pflag.Parse()

	authToken := os.Getenv("SIGNER_TOKEN")
	if authToken == "" {
		fatalf("no env[SIGNER_TOKEN]")
	}

	logger, err := zap.New(zap.KVConfig(log.DebugLevel))
	if err != nil {
		fatalf("create logger: %w", err)
	}

	var srcFile *os.File
	if src == "-" {
		srcFile = os.Stdin
	} else {
		srcFile, err = os.Open(src)
		if err != nil {
			fatalf("open src: %v", err)
		}
		defer func() { _ = srcFile.Close() }()
	}

	var dstFile *os.File
	if dst == "-" {
		dstFile = os.Stdout
	} else {
		dstFile, err = os.Create(dst)
		if err != nil {
			fatalf("open dst: %v", err)
		}
		defer func() { _ = dstFile.Close() }()
	}

	ys, err := signer.NewYaSigner(signer.WithLogger(logger), signer.WithAuthToken(authToken))
	if err != nil {
		fatalf("create signer: %v", err)
	}

	err = ys.Sign(
		context.Background(),
		srcFile,
		dstFile,
		signer.WithSignApplication(app),
		signer.WithSignComment(comment),
	)
	if err != nil {
		fatalf("sign: %v", err)
	}
}
