package simplelog

import (
	"io"
	"sync"
	"time"
)

type (
	Logger struct {
		ctx     []interface{}
		mu      *sync.Mutex
		colored bool
		writer  io.Writer
	}

	LoggerOpts struct {
		Colored bool
		Writer  io.Writer
	}
)

var (
	maxLvl = InfoLevel
)

func NewLogger(opts *LoggerOpts) *Logger {
	return &Logger{
		ctx:     []interface{}{},
		mu:      &sync.Mutex{},
		colored: opts.Colored,
		writer:  opts.Writer,
	}
}

func (l *Logger) Debug(msg string, ctx ...interface{}) {
	l.write(msg, DebugLevel, ctx)
}

func (l *Logger) Info(msg string, ctx ...interface{}) {
	l.write(msg, InfoLevel, ctx)
}

func (l *Logger) Warn(msg string, ctx ...interface{}) {
	l.write(msg, WarnLevel, ctx)
}

func (l *Logger) Error(msg string, ctx ...interface{}) {
	l.write(msg, ErrorLevel, ctx)
}

func (l *Logger) Crit(msg string, ctx ...interface{}) {
	l.write(msg, CritLevel, ctx)
}

func (l *Logger) Raw(msg []byte) {
	l.mu.Lock()
	_, _ = l.writer.Write(msg)
	l.mu.Unlock()
}

func (l *Logger) Child(ctx ...interface{}) *Logger {
	return &Logger{
		ctx:     newContext(l.ctx, ctx),
		mu:      l.mu,
		writer:  l.writer,
		colored: l.colored,
	}
}

func (l *Logger) write(msg string, lvl Lvl, ctx []interface{}) {
	if lvl > maxLvl {
		return
	}

	record := Record{
		Time: time.Now(),
		Lvl:  lvl,
		Msg:  msg,
		Ctx:  newContext(l.ctx, ctx),
	}

	l.mu.Lock()
	_, _ = l.writer.Write(FormatRecord(record, l.colored))
	l.mu.Unlock()
}
