package xlog

import (
	"fmt"
	"net/url"
	"os"
	"path/filepath"

	"go.uber.org/zap"
)

const (
	DefaultXRotateScheme  = "xrotate"
	DefaultXRotateMaxSize = 1024 * 1024 * 10
)

type XRotateSink struct {
	path    string
	file    *os.File
	maxSize int
	curSize int
}

func NewXRotateSink(u *url.URL, opts ...XRotateOption) (zap.Sink, error) {
	if u.Fragment != "" {
		return nil, fmt.Errorf("fragments not allowed with logrotate file URLs: got %v", u)
	}

	if hn := u.Hostname(); hn != "" && hn != "localhost" {
		return nil, fmt.Errorf("logrotate file URLs must leave host empty or use localhost: got %v", u)
	}

	sink := &XRotateSink{
		path:    u.Path[1:],
		maxSize: DefaultXRotateMaxSize,
	}

	for _, opt := range opts {
		switch v := opt.(type) {
		case XRotateOptionMaxSize:
			sink.maxSize = v.maxSize
		}
	}

	if err := sink.reopen(); err != nil {
		return nil, err
	}

	return sink, nil
}

func (s *XRotateSink) reopen() error {
	if err := os.MkdirAll(filepath.Dir(s.path), 0o700); err != nil {
		return fmt.Errorf("failed to create dir for log: %w", err)
	}

	fi, err := os.Stat(s.path)
	switch {
	case err != nil:
		fallthrough
	case fi.Mode()&os.ModeType != 0:
		_ = os.RemoveAll(s.path)
	default:
		s.curSize = int(fi.Size())
	}

	if s.curSize > s.maxSize-1024 {
		return s.rotate()
	}

	return s.openFile()
}

func (s *XRotateSink) rotate() error {
	if s.file != nil {
		_ = s.file.Close()
	}

	if err := os.Rename(s.path, s.path+".old"); err != nil {
		_ = os.RemoveAll(s.path)
	}

	return s.openFile()
}

func (s *XRotateSink) openFile() error {
	file, err := os.OpenFile(s.path, os.O_WRONLY|os.O_APPEND|os.O_CREATE, 0o600)
	if err != nil {
		return fmt.Errorf("failed to open log file: %w", err)
	}

	s.file = file
	s.curSize = 0
	return nil
}

func (s *XRotateSink) Close() error {
	return s.file.Close()
}

func (s *XRotateSink) Write(b []byte) (int, error) {
	n, err := s.file.Write(b)
	if err != nil {
		return n, err
	}

	s.curSize += n
	if s.curSize > s.maxSize {
		return n, s.rotate()
	}
	return n, nil
}

func (s *XRotateSink) Sync() error {
	return s.file.Sync()
}
