package yacsrf

import (
	"bytes"
	"crypto/hmac"
	"crypto/sha1"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"
)

const (
	TokenTTL = 86400
)

type (
	Options struct {
		// SecretKey is used to generate/validate anti-CSRF token
		SecretKey string

		// TTL - token expiry timeout (default: see TokenTTL)
		TTL int64
	}

	YaCsrf struct {
		sk  []byte
		ttl int64
	}
)

func New(opts Options) (*YaCsrf, error) {
	if len(opts.SecretKey) < 10 {
		return nil, fmt.Errorf("secret key length must gather than 10 characters")
	}

	ttl := opts.TTL
	if ttl == 0 {
		ttl = TokenTTL
	}
	return &YaCsrf{
		sk:  []byte(opts.SecretKey),
		ttl: ttl,
	}, nil
}

func (y *YaCsrf) Generate(uid uint64, yandexuid string) (string, error) {
	return y.generate(uid, yandexuid, 0)
}

func (y *YaCsrf) Validate(sk string, uid uint64, yandexuid string) error {
	data := strings.Split(sk, ":")
	if len(data) != 2 {
		return errors.New("wrong format: must be 'token:timestamp'")
	}

	timestamp, err := strconv.ParseInt(data[1], 10, 64)
	if err != nil {
		return fmt.Errorf("timestamp parsing error: %w", err)
	}

	if time.Now().Unix()-timestamp > y.ttl {
		return fmt.Errorf("expired: %d > %d", time.Now().Unix()-timestamp, y.ttl)
	}

	checkSk, err := y.generate(uid, yandexuid, timestamp)
	if err != nil {
		return fmt.Errorf("generate check token failed: %w", err)
	}

	if !hmac.Equal([]byte(sk), []byte(checkSk)) {
		return errors.New("verification failed")
	}
	return nil
}

func (y *YaCsrf) generate(uid uint64, yandexuid string, timestamp int64) (string, error) {
	if timestamp == 0 {
		timestamp = time.Now().Unix()
	}

	mess := bytes.NewBuffer(nil)
	mess.WriteString(strconv.FormatUint(uid, 10))
	mess.WriteByte(':')
	mess.WriteString(yandexuid)
	mess.WriteByte(':')
	mess.WriteString(strconv.FormatInt(timestamp, 10))

	h := hmac.New(sha1.New, y.sk)
	_, err := h.Write(mess.Bytes())
	return fmt.Sprintf("%x:%d", h.Sum(nil), timestamp), err
}
