package yahttp

import (
	"bytes"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"io"
	"net"
	"net/http"
	"net/url"
	"time"

	"a.yandex-team.ru/library/go/certifi"
)

const (
	RedirectNoFollow RedirectPolicy = iota
	RedirectFollow
	RedirectFollowSameOrigin

	GracefulBodyReadBytes = 128 << 10
)

type (
	Client = http.Client

	Config struct {
		// RedirectPolicy allow to specify http client action in case of redirect
		// Default: do not follow
		RedirectPolicy RedirectPolicy

		// DialTimeout is the maximum amount of time a dial will wait for
		// a connect to complete. If Deadline is also set, it may fail
		// earlier.
		DialTimeout time.Duration

		// Timeout specifies a time limit for requests made by this
		// Client. The timeout includes connection time, any
		// redirects, and reading the response body. The timer remains
		// running after Get, Head, Post, or Do return and will
		// interrupt reading of the Response.Body.
		Timeout time.Duration
	}

	RedirectPolicy int
)

var (
	DefaultConfig = Config{
		RedirectPolicy: RedirectNoFollow,
		Timeout:        5 * time.Second,
		DialTimeout:    1 * time.Second,
	}

	DefaultClient = NewClient(DefaultConfig)
)

func NewClient(cfg Config) *Client {
	tlsConfig := new(tls.Config)
	internalCAs, err := certifi.NewCertPool()
	if err == nil {
		tlsConfig.RootCAs = internalCAs
	}

	return &Client{
		Transport: &http.Transport{
			TLSClientConfig: tlsConfig,
			Proxy:           http.ProxyFromEnvironment,
			DialContext: (&net.Dialer{
				Timeout:   cfg.DialTimeout,
				KeepAlive: 60 * time.Second,
			}).DialContext,
			TLSHandshakeTimeout: 2 * time.Second,
		},
		Timeout: cfg.Timeout,
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			switch cfg.RedirectPolicy {
			case RedirectFollow:
				return nil
			case RedirectNoFollow:
				return http.ErrUseLastResponse
			case RedirectFollowSameOrigin:
				if via[len(via)-1].URL.Host != req.URL.Host {
					// disable cross origin redirects
					return http.ErrUseLastResponse
				}
			default:
				return fmt.Errorf("unknown redirection policy: %d", cfg.RedirectPolicy)
			}
			return nil
		},
	}
}

// GracefulClose issues to close HTTP response in safer manner
func GracefulClose(body io.ReadCloser) {
	_, _ = io.CopyN(io.Discard, body, GracefulBodyReadBytes)
	_ = body.Close()
}

// Do sends an HTTP request and returns an HTTP response.
// Any returned error will be of type *url.Error. The url.Error
// value's Timeout method will report true if request timed out or was
// canceled.
//
// DoRequest is a wrapper around DefaultClient.Do
func DoRequest(req *http.Request) (*http.Response, error) {
	return DefaultClient.Do(req)
}

// Get issues a GET to the specified URL.
//
// Get is a wrapper around DefaultClient.Get
func Get(url string) (resp *http.Response, err error) {
	return DefaultClient.Get(url)
}

// Head issues a HEAD to the specified URL.
//
// Head is a wrapper around DefaultClient.Head
func Head(url string) (resp *http.Response, err error) {
	return DefaultClient.Head(url)
}

// Post issues a POST to the specified URL.
//
// Caller should close resp.Body when done reading from it.
//
// Post is a wrapper around DefaultClient.Post.
func Post(url, contentType string, body io.Reader) (resp *http.Response, err error) {
	return DefaultClient.Post(url, contentType, body)
}

// PostForm issues a POST to the specified URL, with data's keys and
// values URL-encoded as the request body.
//
// The Content-Type header is set to application/x-www-form-urlencoded.
//
// PostForm is a wrapper around DefaultClient.PostForm.
func PostForm(url string, data url.Values) (resp *http.Response, err error) {
	return DefaultClient.PostForm(url, data)
}

// PostForm issues a POST to the specified URL, with data's keys and
// values URL-encoded as the request body.
//
// The Content-Type header is set to application/json.
//
// PostForm is a wrapper around DefaultClient.PostForm.
func PostJSON(url string, data interface{}) (resp *http.Response, err error) {
	var jsonBody []byte
	jsonBody, err = json.Marshal(data)
	if err != nil {
		return
	}

	return DefaultClient.Post(url, "application/json; charset=utf-8", bytes.NewBuffer(jsonBody))
}
