package yamake

import (
	"bufio"
	"errors"
	"io"
	"regexp"
	"strings"
)

type (
	tokenizer struct {
		scanner *bufio.Scanner
		tokens  []string
	}

	yMacro struct {
		Name string
		Args []string
	}
)

var (
	tokenRe         = regexp.MustCompile(`(?:"(?:[^"\\]|\\.)*?")|(?:'(?:[^'\\]|\\.)*?')|[^\t\n\f\r ()=]+|\(|\)|=`)
	ErrYaMakeSyntax = errors.New("invalid ya.make syntax")
)

func newTokenizer(r io.Reader) *tokenizer {
	return &tokenizer{
		scanner: bufio.NewScanner(r),
	}
}

func (t *tokenizer) scan() bool {
	if len(t.tokens) != 0 {
		t.tokens = t.tokens[1:]
	}

	for len(t.tokens) == 0 {
		if !t.scanner.Scan() {
			return false
		}

		var comment string
		lineBuffer := t.scanner.Text()
		commentIndex := strings.IndexByte(lineBuffer, '#')
		if commentIndex != -1 {
			comment = lineBuffer[commentIndex:]
			lineBuffer = lineBuffer[:commentIndex]
		}

		t.tokens = tokenRe.FindAllString(lineBuffer, -1)
		if comment != "" {
			t.tokens = append(t.tokens, strings.Trim(comment, " \n"))
		}
	}

	return true
}

func (t *tokenizer) token() string {
	if len(t.tokens) != 0 {
		return t.tokens[0]
	}

	return ""
}

func (t *tokenizer) readMacro() (*yMacro, error) {
	if !t.scan() {
		return nil, t.scanner.Err()
	}

	if t.token() == "(" || t.token() == ")" {
		return nil, ErrYaMakeSyntax
	}

	if t.token()[0] == '#' {
		for {
			if !t.scan() {
				return nil, t.scanner.Err()
			}

			if t.token()[0] != '#' {
				break
			}
		}
	}

	m := &yMacro{}
	m.Name = t.token()

	if !t.scan() || t.token() != "(" {
		return nil, ErrYaMakeSyntax
	}

	for {
		if !t.scan() {
			return nil, t.scanner.Err()
		}

		if t.token() == ")" {
			return m, nil
		}

		m.Args = append(m.Args, t.token())
	}
}
